/**
 * common.js ユニットテスト
 */

import { describe, it, expect } from 'vitest';
import { escapeHtml, extractTrackingNumbers, isPhoneNumber, isValidTrackingNumber, getStatusType } from './common.js';

describe('escapeHtml', () => {
  it('空文字列をエスケープ', () => {
    expect(escapeHtml('')).toBe('');
  });

  it('特殊文字をエスケープ', () => {
    expect(escapeHtml('<script>alert("XSS")</script>')).toBe('&lt;script&gt;alert(&quot;XSS&quot;)&lt;/script&gt;');
  });

  it('アンパサンドをエスケープ', () => {
    expect(escapeHtml('Tom & Jerry')).toBe('Tom &amp; Jerry');
  });

  it('シングルクォートをエスケープ', () => {
    expect(escapeHtml("It's a test")).toBe('It&#039;s a test');
  });

  it('複数の特殊文字をエスケープ', () => {
    expect(escapeHtml('<div>Test & "quote" & \'single\'</div>'))
      .toBe('&lt;div&gt;Test &amp; &quot;quote&quot; &amp; &#039;single&#039;&lt;/div&gt;');
  });

  it('null/undefinedの場合は空文字列を返す', () => {
    expect(escapeHtml(null)).toBe('');
    expect(escapeHtml(undefined)).toBe('');
  });

  it('文字列以外は空文字列を返す', () => {
    expect(escapeHtml(123)).toBe('');
    expect(escapeHtml({})).toBe('');
    expect(escapeHtml([])).toBe('');
  });
});

describe('extractTrackingNumbers', () => {
  it('空文字列の場合は空配列を返す', () => {
    expect(extractTrackingNumbers('')).toEqual([]);
  });

  it('10桁の数字を抽出', () => {
    expect(extractTrackingNumbers('1234567890')).toEqual(['1234567890']);
  });

  it('12桁の数字を抽出', () => {
    expect(extractTrackingNumbers('123456789012')).toEqual(['123456789012']);
  });

  it('カンマ区切りで複数抽出', () => {
    expect(extractTrackingNumbers('1234567890,9876543210'))
      .toEqual(['1234567890', '9876543210']);
  });

  it('改行区切りで複数抽出', () => {
    expect(extractTrackingNumbers('1234567890\n9876543210'))
      .toEqual(['1234567890', '9876543210']);
  });

  it('スペース区切りで複数抽出', () => {
    expect(extractTrackingNumbers('1234567890 9876543210'))
      .toEqual(['1234567890', '9876543210']);
  });

  it('9桁以下は除外', () => {
    expect(extractTrackingNumbers('123456789')).toEqual([]);
  });

  it('13桁以上は除外', () => {
    expect(extractTrackingNumbers('1234567890123')).toEqual([]);
  });

  it('重複を除去', () => {
    expect(extractTrackingNumbers('1234567890,1234567890,9876543210'))
      .toEqual(['1234567890', '9876543210']);
  });

  it('混合フォーマットに対応', () => {
    expect(extractTrackingNumbers('1234567890, 9876543210\n1111111111'))
      .toEqual(['1234567890', '9876543210', '1111111111']);
  });

  it('null/undefinedの場合は空配列を返す', () => {
    expect(extractTrackingNumbers(null)).toEqual([]);
    expect(extractTrackingNumbers(undefined)).toEqual([]);
  });

  it('文字列以外は空配列を返す', () => {
    expect(extractTrackingNumbers(123)).toEqual([]);
    expect(extractTrackingNumbers({})).toEqual([]);
    expect(extractTrackingNumbers([])).toEqual([]);
  });
});

describe('isPhoneNumber', () => {
  it('固定電話番号を検出（03）', () => {
    expect(isPhoneNumber('0312345678')).toBe(true);
  });

  it('固定電話番号を検出（06）', () => {
    expect(isPhoneNumber('0612345678')).toBe(true);
  });

  it('フリーダイヤルを検出（0120）', () => {
    expect(isPhoneNumber('0120123456')).toBe(true);
  });

  it('ナビダイヤルを検出（0570）', () => {
    expect(isPhoneNumber('0570123456')).toBe(true);
  });

  it('携帯電話番号を検出', () => {
    expect(isPhoneNumber('09012345678')).toBe(true);
    expect(isPhoneNumber('08012345678')).toBe(true);
    expect(isPhoneNumber('07012345678')).toBe(true);
  });

  it('10桁の数字以外はfalse', () => {
    expect(isPhoneNumber('123456789')).toBe(false);
    expect(isPhoneNumber('')).toBe(false);
    expect(isPhoneNumber('abcdefghij')).toBe(false);
  });
});

describe('isValidTrackingNumber', () => {
  it('10桁の数字は有効', () => {
    expect(isValidTrackingNumber('1234567890')).toBe(true);
  });

  it('12桁の数字は有効', () => {
    expect(isValidTrackingNumber('123456789012')).toBe(true);
  });

  it('電話番号は無効', () => {
    expect(isValidTrackingNumber('0312345678')).toBe(false);
    expect(isValidTrackingNumber('09012345678')).toBe(false);
  });

  it('9桁以下は無効', () => {
    expect(isValidTrackingNumber('123456789')).toBe(false);
  });

  it('13桁以上は無効', () => {
    expect(isValidTrackingNumber('1234567890123')).toBe(false);
  });

  it('数字以外は無効', () => {
    expect(isValidTrackingNumber('abcdefghij')).toBe(false);
    expect(isValidTrackingNumber('')).toBe(false);
  });
});

describe('getStatusType', () => {
  it('完了ステータスを判定', () => {
    expect(getStatusType('配達完了')).toBe('delivered');
    expect(getStatusType('到着完了')).toBe('delivered');
    expect(getStatusType('お届け完了')).toBe('delivered');
  });

  it('配達中ステータスを判定', () => {
    expect(getStatusType('配達中')).toBe('intransit');
    expect(getStatusType('持出')).toBe('intransit');
    expect(getStatusType('配達持出')).toBe('intransit');
  });

  it('発送ステータスを判定', () => {
    expect(getStatusType('発送済み')).toBe('shipped');
    expect(getStatusType('出荷')).toBe('shipped');
  });

  it('受付ステータスを判定', () => {
    expect(getStatusType('受付済み')).toBe('pickedup');
    expect(getStatusType('集荷済み')).toBe('pickedup');
  });

  it('不明なステータスはunknown', () => {
    expect(getStatusType('不明')).toBe('unknown');
    expect(getStatusType('')).toBe('unknown');
    expect(getStatusType(undefined)).toBe('unknown');
    expect(getStatusType(null)).toBe('unknown');
  });
});
