/**
 * 西濃追跡チェッカー - 共通ユーティリティ関数
 *
 * Chrome拡張機能のService WorkerとContent Scriptで使用するため、
 * ES6モジュール構文ではなく、グローバルスコープに関数を定義します。
 */

/**
 * HTMLをエスケープする（XSS対策）
 * @param {string} unsafe - エスケープ対象の文字列
 * @returns {string} エスケープされた文字列
 */
function escapeHtml(unsafe) {
  if (typeof unsafe !== 'string') {
    return '';
  }
  return unsafe
    .replace(/&/g, '&amp;')
    .replace(/</g, '&lt;')
    .replace(/>/g, '&gt;')
    .replace(/"/g, '&quot;')
    .replace(/'/g, '&#039;');
}

/**
 * 伝票番号を抽出する
 * @param {string} text - テキスト（カンマ区切り、改行区切り、スペース区切り対応）
 * @returns {string[]} 伝票番号の配列（重複なし）
 */
function extractTrackingNumbers(text) {
  if (typeof text !== 'string') {
    return [];
  }
  // カンマ区切り、改行区切り、スペース区切りに対応
  // 10-12桁の数字を伝票番号として抽出
  const numbers = text
    .split(/[,\n\s]+/)
    .map(n => n.trim())
    .filter(n => /^\d{10,12}$/.test(n));

  return [...new Set(numbers)]; // 重複を削除
}

/**
 * 電話番号かどうかを判定
 * @param {string} num - チェックする数字文字列
 * @returns {boolean} 電話番号の場合はtrue
 */
function isPhoneNumber(num) {
  // 電話番号のパターン（0から始まる）
  // 03, 06, 052, 0120, 0570, 0800 など
  const phonePatterns = [
    /^0[1-9]\d{8,9}$/,  // 固定電話（03, 06, 052など）
    /^0120\d{6}$/,       // フリーダイヤル
    /^0570\d{6}$/,       // ナビダイヤル
    /^0800\d{7}$/,       // フリーダイヤル
  ];

  return phonePatterns.some(pattern => pattern.test(num));
}

/**
 * 伝票番号として有効かどうかを判定
 * @param {string} num - チェックする数字文字列
 * @returns {boolean} 有効な伝票番号の場合はtrue
 */
function isValidTrackingNumber(num) {
  // 10-12桁の数字
  if (!/^\d{10,12}$/.test(num)) {
    return false;
  }

  // 電話番号は除外
  if (isPhoneNumber(num)) {
    return false;
  }

  return true;
}

/**
 * ステータスタイプの判定
 * @param {string} status - ステータス文字列
 * @returns {string} ステータスタイプ（delivered, intransit, shipped, pickedup, unknown）
 */
function getStatusType(status) {
  if (!status) return 'unknown';

  if (status.includes('完了') || status.includes('到着') || status.includes('お届け')) {
    return 'delivered';
  }
  if (status.includes('持出') || status.includes('配達中')) {
    return 'intransit';
  }
  if (status.includes('発送') || status.includes('出荷')) {
    return 'shipped';
  }
  if (status.includes('受付') || status.includes('集荷')) {
    return 'pickedup';
  }

  return 'unknown';
}
