/**
 * popup.js ユニットテスト
 *
 * 注意: このテストファイルはpopup.jsの動作を検証するため、
 * テスト内でinnerHTMLをモックしています。実際のXSS脆弱性を作成するものではありません。
 */

import { describe, it, expect, vi, beforeEach, afterEach } from 'vitest';
import { escapeHtml } from '../utils/common.js';

// DOMモックのセットアップ
function setupDOMMock() {
  const mockElements = {
    'numbers-input': { value: '', addEventListener: vi.fn() },
    'track-btn': {
      disabled: false,
      textContent: '追跡する',
      addEventListener: vi.fn()
    },
    'history-section': { style: { display: 'block' }, addEventListener: vi.fn() },
    'history': {
      // innerHTMLモック（セキュリティ検証のため、テスト内でのみ使用）
      _innerHTML: '',
      get innerHTML() { return this._innerHTML; },
      set innerHTML(value) { this._innerHTML = value; },
      textContent: '',
      appendChild: vi.fn()
    },
    'clear-history-btn': { addEventListener: vi.fn() },
    'extract-section': { style: { display: 'none' }, addEventListener: vi.fn() },
    'extract-btn': { addEventListener: vi.fn() },
    'extract-count': { textContent: '', addEventListener: vi.fn() }
  };

  global.document = {
    getElementById: vi.fn((id) => mockElements[id]),
    addEventListener: vi.fn(),
    createElement: vi.fn((tag) => {
      const element = {
        tagName: tag,
        className: '',
        style: { cssText: '' },
        textContent: '',
        _innerHTML: '',
        get innerHTML() { return this._innerHTML; },
        set innerHTML(value) { this._innerHTML = value; },
        appendChild: vi.fn()
      };
      return element;
    })
  };

  // alertとconfirmをモック
  global.alert = vi.fn();
  global.confirm = vi.fn();

  return mockElements;
}

// Chrome APIモックのセットアップ
function setupChromeMock() {
  let storageData = {};

  const mockStorage = {
    local: {
      get: vi.fn((keys) => {
        const result = {};
        if (keys.includes('history') && storageData.history) {
          result.history = storageData.history;
        }
        if (keys.includes('latestResults') && storageData.latestResults) {
          result.latestResults = storageData.latestResults;
        }
        if (keys.includes('extractedNumbers') && storageData.extractedNumbers) {
          result.extractedNumbers = storageData.extractedNumbers;
        }
        return Promise.resolve(result);
      }),
      set: vi.fn((data) => {
        Object.assign(storageData, data);
        return Promise.resolve();
      }),
      remove: vi.fn((keys) => {
        keys.forEach(key => delete storageData[key]);
        return Promise.resolve();
      })
    }
  };

  const mockTabs = {
    query: vi.fn(() => Promise.resolve([{ id: 1, url: 'https://example.com' }])),
    sendMessage: vi.fn(() => Promise.resolve({ numbers: ['1234567890'] }))
  };

  const mockScripting = {
    executeScript: vi.fn(() => Promise.resolve())
  };

  const mockRuntime = {
    sendMessage: vi.fn(() => Promise.resolve({
      success: true,
      results: [
        { number: '1234567890', status: '配達完了', statusType: 'delivered', timestamp: '2026-01-23T00:00:00Z' }
      ]
    }))
  };

  global.chrome = {
    storage: mockStorage,
    tabs: mockTabs,
    scripting: mockScripting,
    runtime: mockRuntime
  };

  return { mockStorage, mockTabs, mockScripting, mockRuntime, getStorageData: () => storageData };
}

// テスト対象関数の定義（popup.jsから抽出）
function createTestFunctions(mockElements, mockStorage) {
  // 最新の結果を履歴に保存
  async function loadLatestResults() {
    try {
      const result = await mockStorage.local.get(['latestResults']);
      if (result.latestResults && result.latestResults.length > 0) {
        await saveToHistory(result.latestResults);
        await mockStorage.local.remove(['latestResults']);
      }
    } catch (error) {
      console.error('Error loading latest results:', error);
    }
  }

  // 履歴に保存
  async function saveToHistory(results) {
    try {
      const result = await mockStorage.local.get(['history']);
      const history = result.history || [];

      const successfulResults = results.filter(item => {
        if (item.error) {
          return false;
        }
        if (item.status === '不明') {
          return false;
        }
        return true;
      });

      if (successfulResults.length === 0) {
        return;
      }

      const newNumbers = new Set(successfulResults.map(r => r.number));
      const filteredHistory = history.filter(h => !newNumbers.has(h.number));
      const newHistory = [...successfulResults, ...filteredHistory];
      const limitedHistory = newHistory.slice(0, 50);

      await mockStorage.local.set({ history: limitedHistory });
      loadHistory();
    } catch (error) {
      console.error('Error saving to history:', error);
    }
  }

  // 履歴を表示
  async function loadHistory() {
    try {
      const result = await mockStorage.local.get(['history']);
      mockElements['history-section'].style.display = 'block';

      if (!result.history || result.history.length === 0) {
        mockElements['history'].innerHTML = `
          <div class="empty-state" style="padding: 20px; text-align: center; color: #999;">
            <p>📜 履歴はありません</p>
            <p style="font-size: 11px; margin-top: 8px;">※追跡成功したものだけ保存されます</p>
          </div>
        `;
        return;
      }

      const html = result.history.map((item) => {
        const statusClass = item.statusType || 'unknown';
        const timestamp = item.timestamp
          ? new Date(item.timestamp).toLocaleString('ja-JP')
          : '日時なし';

        return `
          <div class="result-item status-${statusClass}" style="padding: 10px; margin-bottom: 8px;">
            <div style="margin-bottom: 4px;">
              <span style="color: #666; font-size: 10px;">お問い合わせ番号</span>
              <div style="font-weight: bold; font-size: 13px;">${escapeHtml(item.number)}</div>
            </div>
            <div style="margin-bottom: 4px;">
              <span style="color: #666; font-size: 10px;">お届け状況</span>
              <div style="font-size: 12px;">${escapeHtml(item.status || '不明')}</div>
            </div>
            <div style="font-size: 10px; color: #999;">${escapeHtml(timestamp)}</div>
          </div>
        `;
      }).join('');

      mockElements['history'].innerHTML = html;
    } catch (error) {
      console.error('Error loading history:', error);
      mockElements['history'].textContent = '';
      const errorDiv = global.document.createElement('div');
      errorDiv.className = 'empty-state';
      errorDiv.style.cssText = 'padding: 20px; text-align: center; color: #dc3545;';
      errorDiv.innerHTML = '<p>履歴の読み込みに失敗しました</p>';
      const errorMsg = global.document.createElement('p');
      errorMsg.style.cssText = 'font-size: 11px; margin-top: 8px;';
      errorMsg.textContent = error.message;
      errorDiv.appendChild(errorMsg);
      mockElements['history'].appendChild(errorDiv);
    }
  }

  // 履歴をクリア
  async function handleClearHistory() {
    if (!global.confirm()) {
      return;
    }

    try {
      await mockStorage.local.remove(['history']);
      mockElements['history'].innerHTML = '';
      mockElements['history-section'].style.display = 'none';
    } catch (error) {
      console.error('Error clearing history:', error);
    }
  }

  // ページから抽出可能な伝票番号をチェック
  async function checkExtractableNumbers() {
    try {
      const [tab] = await global.chrome.tabs.query({ active: true, currentWindow: true });

      if (!tab || tab.url.startsWith('chrome://')) {
        return;
      }

      const response = await global.chrome.tabs.sendMessage(tab.id, {
        action: 'extractNumbersFromPage'
      });

      if (response && response.numbers && response.numbers.length > 0) {
        mockElements['extract-section'].style.display = 'flex';
        mockElements['extract-count'].textContent = `(${response.numbers.length}件)`;
        await mockStorage.local.set({ extractedNumbers: response.numbers });
      } else {
        mockElements['extract-section'].style.display = 'none';
      }
    } catch (error) {
      mockElements['extract-section'].style.display = 'none';
    }
  }

  // 抽出ボタンクリック時の処理
  async function handleExtract() {
    try {
      const result = await mockStorage.local.get(['extractedNumbers']);

      if (!result.extractedNumbers || result.extractedNumbers.length === 0) {
        global.alert('抽出された伝票番号がありません');
        return;
      }

      mockElements['numbers-input'].value = result.extractedNumbers.join('\n');
      mockElements['extract-section'].style.display = 'none';
    } catch (error) {
      global.alert(`エラーが発生しました: ${error.message}`);
    }
  }

  // 追跡ボタンクリック時の処理
  async function handleTrack() {
    const text = mockElements['numbers-input'].value.trim();

    if (!text) {
      global.alert('伝票番号を入力してください');
      return;
    }

    mockElements['track-btn'].disabled = true;
    mockElements['track-btn'].textContent = '追跡中...';

    try {
      const response = await global.chrome.runtime.sendMessage({
        action: 'trackNumbers',
        numbers: [text]
      });

      if (response.success) {
        await saveToHistory(response.results);
        mockElements['numbers-input'].value = '';
      } else {
        global.alert(`エラー: ${response.error}`);
      }
    } catch (error) {
      global.alert(`エラーが発生しました: ${error.message}`);
    } finally {
      mockElements['track-btn'].disabled = false;
      mockElements['track-btn'].textContent = '追跡する';
    }
  }

  return {
    loadLatestResults,
    saveToHistory,
    loadHistory,
    handleClearHistory,
    checkExtractableNumbers,
    handleExtract,
    handleTrack
  };
}

describe('popup.js', () => {
  let mockElements;
  let chromeMocks;
  let testFunctions;

  beforeEach(() => {
    mockElements = setupDOMMock();
    chromeMocks = setupChromeMock();
    vi.spyOn(console, 'error').mockImplementation(() => {});
    testFunctions = createTestFunctions(mockElements, chromeMocks.mockStorage);
  });

  afterEach(() => {
    vi.restoreAllMocks();
  });

  describe('saveToHistory', () => {
    it('正常な追跡結果を履歴に保存', async () => {
      const results = [
        { number: '1234567890', status: '配達完了', statusType: 'delivered', timestamp: '2026-01-23T00:00:00Z' }
      ];

      await testFunctions.saveToHistory(results);

      expect(chromeMocks.mockStorage.local.set).toHaveBeenCalledWith(
        expect.objectContaining({
          history: expect.arrayContaining([
            expect.objectContaining({ number: '1234567890' })
          ])
        })
      );
    });

    it('エラーありの結果は履歴に保存しない', async () => {
      const results = [
        { number: '1234567890', error: 'タブが閉じられました' }
      ];

      await testFunctions.saveToHistory(results);

      expect(chromeMocks.mockStorage.local.set).not.toHaveBeenCalled();
    });

    it('ステータス「不明」の結果は履歴に保存しない', async () => {
      const results = [
        { number: '1234567890', status: '不明', statusType: 'unknown' }
      ];

      await testFunctions.saveToHistory(results);

      expect(chromeMocks.mockStorage.local.set).not.toHaveBeenCalled();
    });

    it('重複する伝票番号は新しい結果で上書き', async () => {
      chromeMocks.getStorageData().history = [
        { number: '1234567890', status: '配達中', statusType: 'intransit', timestamp: '2026-01-22T00:00:00Z' }
      ];

      const results = [
        { number: '1234567890', status: '配達完了', statusType: 'delivered', timestamp: '2026-01-23T00:00:00Z' }
      ];

      await testFunctions.saveToHistory(results);

      const setCall = chromeMocks.mockStorage.local.set.mock.calls[0][0];
      expect(setCall.history).toHaveLength(1);
      expect(setCall.history[0].status).toBe('配達完了');
    });

    it('履歴を最大50件に制限', async () => {
      chromeMocks.getStorageData().history = Array.from({ length: 50 }, (_, i) => ({
        number: `${1000000000 + i}`,
        status: '配達完了',
        statusType: 'delivered',
        timestamp: '2026-01-22T00:00:00Z'
      }));

      const results = Array.from({ length: 5 }, (_, i) => ({
        number: `200000000${i}`,
        status: '配達完了',
        statusType: 'delivered',
        timestamp: '2026-01-23T00:00:00Z'
      }));

      await testFunctions.saveToHistory(results);

      const setCall = chromeMocks.mockStorage.local.set.mock.calls[0][0];
      expect(setCall.history).toHaveLength(50);
    });

    it('成功結果が0件の場合は保存しない', async () => {
      const results = [
        { number: '1234567890', error: 'タブが閉じられました' },
        { number: '9876543210', status: '不明', statusType: 'unknown' }
      ];

      await testFunctions.saveToHistory(results);

      expect(chromeMocks.mockStorage.local.set).not.toHaveBeenCalled();
    });

    it('複数の成功結果を履歴に保存', async () => {
      const results = [
        { number: '1234567890', status: '配達完了', statusType: 'delivered', timestamp: '2026-01-23T00:00:00Z' },
        { number: '9876543210', status: '配達中', statusType: 'intransit', timestamp: '2026-01-23T01:00:00Z' }
      ];

      await testFunctions.saveToHistory(results);

      const setCall = chromeMocks.mockStorage.local.set.mock.calls[0][0];
      expect(setCall.history).toHaveLength(2);
    });
  });

  describe('loadHistory', () => {
    it('履歴がない場合は空状態メッセージを表示', async () => {
      await testFunctions.loadHistory();

      expect(mockElements['history'].innerHTML).toContain('履歴はありません');
    });

    it('履歴がある場合はHTMLを生成', async () => {
      chromeMocks.getStorageData().history = [
        { number: '1234567890', status: '配達完了', statusType: 'delivered', timestamp: '2026-01-23T00:00:00Z' }
      ];

      await testFunctions.loadHistory();

      expect(mockElements['history'].innerHTML).toContain('1234567890');
      expect(mockElements['history'].innerHTML).toContain('配達完了');
    });

    it('HTMLエスケープを適用（XSS対策）', async () => {
      chromeMocks.getStorageData().history = [
        { number: '<script>alert("XSS")</script>', status: '<img>test', statusType: 'delivered', timestamp: '2026-01-23T00:00:00Z' }
      ];

      await testFunctions.loadHistory();

      // escapeHtml関数が適用されていることを確認
      expect(mockElements['history'].innerHTML).toContain('&lt;script&gt;alert(&quot;XSS&quot;)&lt;/script&gt;');
      expect(mockElements['history'].innerHTML).toContain('&lt;img&gt;test');
      // 生の危険なHTMLタグが含まれていないことを確認
      expect(mockElements['history'].innerHTML).not.toContain('<script>alert("XSS")</script>');
      expect(mockElements['history'].innerHTML).not.toContain('<img>test');
    });

    it('タイムスタンプを日本語フォーマットで表示', async () => {
      chromeMocks.getStorageData().history = [
        { number: '1234567890', status: '配達完了', statusType: 'delivered', timestamp: '2026-01-23T12:34:56Z' }
      ];

      await testFunctions.loadHistory();

      expect(mockElements['history'].innerHTML).toMatch(/\d{4}\/\d{1,2}\/\d{1,2}/);
    });

    it('履歴セクションを常に表示', async () => {
      await testFunctions.loadHistory();

      expect(mockElements['history-section'].style.display).toBe('block');
    });

    it('エラー時に安全なエラーメッセージを表示', async () => {
      chromeMocks.mockStorage.local.get = vi.fn(() => Promise.reject(new Error('Storage error')));

      await testFunctions.loadHistory();

      // textContentがクリアされ、appendChildが呼ばれることを確認
      expect(mockElements['history'].textContent).toBe('');
      expect(mockElements['history'].appendChild).toHaveBeenCalled();
      expect(console.error).toHaveBeenCalledWith('Error loading history:', expect.any(Error));
    });
  });

  describe('handleClearHistory', () => {
    it('確認ダイアログでOKの場合は履歴をクリア', async () => {
      global.confirm.mockReturnValue(true);

      await testFunctions.handleClearHistory();

      expect(chromeMocks.mockStorage.local.remove).toHaveBeenCalledWith(['history']);
      expect(mockElements['history'].innerHTML).toBe('');
      expect(mockElements['history-section'].style.display).toBe('none');
    });

    it('確認ダイアログでキャンセルの場合は何もしない', async () => {
      global.confirm.mockReturnValue(false);

      await testFunctions.handleClearHistory();

      expect(chromeMocks.mockStorage.local.remove).not.toHaveBeenCalled();
    });
  });

  describe('loadLatestResults', () => {
    it('latestResultsがある場合は履歴に保存してクリア', async () => {
      const latestResults = [
        { number: '1234567890', status: '配達完了', statusType: 'delivered', timestamp: '2026-01-23T00:00:00Z' }
      ];
      chromeMocks.getStorageData().latestResults = latestResults;

      await testFunctions.loadLatestResults();

      expect(chromeMocks.mockStorage.local.remove).toHaveBeenCalledWith(['latestResults']);
      expect(chromeMocks.mockStorage.local.set).toHaveBeenCalled();
    });

    it('latestResultsがない場合は何もしない', async () => {
      await testFunctions.loadLatestResults();

      expect(chromeMocks.mockStorage.local.set).not.toHaveBeenCalled();
      expect(chromeMocks.mockStorage.local.remove).not.toHaveBeenCalled();
    });
  });

  describe('checkExtractableNumbers', () => {
    it('抽出可能な伝票番号がある場合は抽出ボタンを表示', async () => {
      await testFunctions.checkExtractableNumbers();

      expect(mockElements['extract-section'].style.display).toBe('flex');
      expect(mockElements['extract-count'].textContent).toBe('(1件)');
    });

    it('抽出した番号をStorageに保存', async () => {
      await testFunctions.checkExtractableNumbers();

      expect(chromeMocks.mockStorage.local.set).toHaveBeenCalledWith(
        expect.objectContaining({
          extractedNumbers: expect.any(Array)
        })
      );
    });

    it('chrome://ページでは何もしない', async () => {
      chromeMocks.mockTabs.query = vi.fn(() => Promise.resolve([{ id: 1, url: 'chrome://extensions' }]));

      await testFunctions.checkExtractableNumbers();

      expect(mockElements['extract-section'].style.display).toBe('none');
    });

    it('エラー時は抽出ボタンを非表示', async () => {
      chromeMocks.mockTabs.sendMessage = vi.fn(() => Promise.reject(new Error('Content script not found')));

      await testFunctions.checkExtractableNumbers();

      expect(mockElements['extract-section'].style.display).toBe('none');
    });

    it('伝票番号がない場合は抽出ボタンを非表示', async () => {
      chromeMocks.mockTabs.sendMessage = vi.fn(() => Promise.resolve({ numbers: [] }));

      await testFunctions.checkExtractableNumbers();

      expect(mockElements['extract-section'].style.display).toBe('none');
    });
  });

  describe('handleExtract', () => {
    it('抽出された番号を入力フィールドにセット', async () => {
      chromeMocks.getStorageData().extractedNumbers = ['1234567890', '9876543210'];

      await testFunctions.handleExtract();

      expect(mockElements['numbers-input'].value).toBe('1234567890\n9876543210');
      expect(mockElements['extract-section'].style.display).toBe('none');
    });

    it('抽出された番号がない場合はアラートを表示', async () => {
      await testFunctions.handleExtract();

      expect(global.alert).toHaveBeenCalledWith('抽出された伝票番号がありません');
    });
  });

  describe('handleTrack', () => {
    it('伝票番号が入力されていない場合はアラートを表示', async () => {
      await testFunctions.handleTrack();

      expect(global.alert).toHaveBeenCalledWith('伝票番号を入力してください');
    });

    it('追跡成功時は履歴に保存して入力欄をクリア', async () => {
      mockElements['numbers-input'].value = '1234567890';

      await testFunctions.handleTrack();

      expect(mockElements['numbers-input'].value).toBe('');
      expect(chromeMocks.mockStorage.local.set).toHaveBeenCalled();
    });

    it('追跡失敗時はエラーアラートを表示', async () => {
      mockElements['numbers-input'].value = '1234567890';
      chromeMocks.mockRuntime.sendMessage = vi.fn(() => Promise.resolve({
        success: false,
        error: 'ネットワークエラー'
      }));

      await testFunctions.handleTrack();

      expect(global.alert).toHaveBeenCalledWith('エラー: ネットワークエラー');
    });

    it('処理完了後にボタンを再有効化', async () => {
      mockElements['numbers-input'].value = '1234567890';

      await testFunctions.handleTrack();

      expect(mockElements['track-btn'].disabled).toBe(false);
      expect(mockElements['track-btn'].textContent).toBe('追跡する');
    });
  });
});
