/**
 * 西濃追跡チェッカー - Popup Script
 *
 * 機能:
 * - Storageから最新の結果を取得して表示
 * - 手動入力の場合: Backgroundにメッセージ送信
 * - 履歴の表示・削除
 *
 * 注意: common.jsはpopup.htmlで先に読み込まれているため、
 * ここではimport文を使用せず、グローバルスコープの関数を直接使用します。
 */

// DOM要素
const numbersInput = document.getElementById('numbers-input');
const trackBtn = document.getElementById('track-btn');
const historySection = document.getElementById('history-section');
const historyDiv = document.getElementById('history');
const clearHistoryBtn = document.getElementById('clear-history-btn');
const extractSection = document.getElementById('extract-section');
const extractBtn = document.getElementById('extract-btn');
const extractCount = document.getElementById('extract-count');

// 初期化
document.addEventListener('DOMContentLoaded', () => {
  loadLatestResults();
  loadHistory();
  setupEventListeners();
  checkExtractableNumbers();
});

// イベントリスナー設定
function setupEventListeners() {
  trackBtn.addEventListener('click', handleTrack);
  clearHistoryBtn.addEventListener('click', handleClearHistory);
  extractBtn.addEventListener('click', handleExtract);
}

// 追跡ボタンクリック時の処理
async function handleTrack() {
  const text = numbersInput.value.trim();

  if (!text) {
    alert('伝票番号を入力してください');
    return;
  }

  // 伝票番号を抽出
  const numbers = extractTrackingNumbers(text);

  if (numbers.length === 0) {
    alert('有効な伝票番号が見つかりませんでした（10-12桁の数字）');
    return;
  }

  // ボタンを無効化
  trackBtn.disabled = true;
  trackBtn.textContent = '追跡中...';

  try {
    // Backgroundにメッセージ送信
    const response = await chrome.runtime.sendMessage({
      action: 'trackNumbers',
      numbers: numbers
    });

    if (response.success) {
      await saveToHistory(response.results);
      // 入力欄をクリア
      numbersInput.value = '';
    } else {
      alert(`エラー: ${response.error}`);
    }
  } catch (error) {
    alert(`エラーが発生しました: ${error.message}`);
  } finally {
    // ボタンを再有効化
    trackBtn.disabled = false;
    trackBtn.textContent = '追跡する';
  }
}

// 最新の結果を履歴に保存
async function loadLatestResults() {
  try {
    const result = await chrome.storage.local.get(['latestResults']);
    if (result.latestResults && result.latestResults.length > 0) {
      // 履歴に保存
      await saveToHistory(result.latestResults);
      // 保存後にlatestResultsをクリア（重複保存を防ぐ）
      await chrome.storage.local.remove(['latestResults']);
    }
  } catch (error) {
    console.error('Error loading latest results:', error);
  }
}

// 履歴に保存
async function saveToHistory(results) {
  try {
    const result = await chrome.storage.local.get(['history']);
    const history = result.history || [];

    // 成功したものだけを履歴に保存（ステータスが「不明」以外）
    const successfulResults = results.filter(item => {
      // エラーがあるものは除外
      if (item.error) {
        return false;
      }
      // ステータスが「不明」のものは除外
      if (item.status === '不明') {
        return false;
      }
      return true;
    });

    if (successfulResults.length === 0) {
      return;
    }

    // 新しい結果の伝票番号セット
    const newNumbers = new Set(successfulResults.map(r => r.number));

    // 既存履歴から重複を除去（新しい結果で上書きするため）
    const filteredHistory = history.filter(h => !newNumbers.has(h.number));

    // 新しい結果を先頭に追加（最新が上）
    const newHistory = [...successfulResults, ...filteredHistory];

    // 最大50件に制限
    const limitedHistory = newHistory.slice(0, 50);

    await chrome.storage.local.set({ history: limitedHistory });

    // 履歴を再表示
    loadHistory();
  } catch (error) {
    console.error('Error saving to history:', error);
  }
}

// 履歴を表示
async function loadHistory() {
  try {
    const result = await chrome.storage.local.get(['history']);

    // 履歴セクションを常に表示
    historySection.style.display = 'block';

    if (!result.history || result.history.length === 0) {
      // 履歴がない場合のメッセージ
      historyDiv.innerHTML = `
        <div class="empty-state" style="padding: 20px; text-align: center; color: #999;">
          <p>📜 履歴はありません</p>
          <p style="font-size: 11px; margin-top: 8px;">※追跡成功したものだけ保存されます</p>
        </div>
      `;
      return;
    }

    const html = result.history.map((item) => {
      const statusClass = item.statusType || 'unknown';
      const timestamp = item.timestamp
        ? new Date(item.timestamp).toLocaleString('ja-JP')
        : '日時なし';

      return `
        <div class="result-item status-${statusClass}" style="padding: 10px; margin-bottom: 8px;">
          <div style="margin-bottom: 4px;">
            <span style="color: #666; font-size: 10px;">お問い合わせ番号</span>
            <div style="font-weight: bold; font-size: 13px;">${escapeHtml(item.number)}</div>
          </div>
          <div style="margin-bottom: 4px;">
            <span style="color: #666; font-size: 10px;">お届け状況</span>
            <div style="font-size: 12px;">${escapeHtml(item.status || '不明')}</div>
          </div>
          <div style="font-size: 10px; color: #999;">${escapeHtml(timestamp)}</div>
        </div>
      `;
    }).join('');

    historyDiv.innerHTML = html;
  } catch (error) {
    console.error('Error loading history:', error);
    // 安全にエラーメッセージを表示（textContent使用）
    historyDiv.textContent = '';
    const errorDiv = document.createElement('div');
    errorDiv.className = 'empty-state';
    errorDiv.style.cssText = 'padding: 20px; text-align: center; color: #dc3545;';
    errorDiv.innerHTML = '<p>履歴の読み込みに失敗しました</p>';
    const errorMsg = document.createElement('p');
    errorMsg.style.cssText = 'font-size: 11px; margin-top: 8px;';
    errorMsg.textContent = error.message;
    errorDiv.appendChild(errorMsg);
    historyDiv.appendChild(errorDiv);
  }
}

// 履歴をクリア
async function handleClearHistory() {
  if (confirm('履歴をクリアしますか？')) {
    try {
      await chrome.storage.local.remove(['history']);
      historyDiv.innerHTML = '';
      historySection.style.display = 'none';
    } catch (error) {
      console.error('Error clearing history:', error);
    }
  }
}

// ページから抽出可能な伝票番号をチェック
async function checkExtractableNumbers() {
  try {
    // アクティブなタブを取得
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

    if (!tab || tab.url.startsWith('chrome://')) {
      return; // Chromeの内部ページでは無効
    }

    // Content Scriptにメッセージ送信して伝票番号を抽出
    const response = await chrome.tabs.sendMessage(tab.id, {
      action: 'extractNumbersFromPage'
    });

    if (response && response.numbers && response.numbers.length > 0) {
      // 抽出ボタンを表示
      extractSection.style.display = 'flex';
      extractCount.textContent = `(${response.numbers.length}件)`;

      // 抽出した番号をStorageに保存
      await chrome.storage.local.set({ extractedNumbers: response.numbers });
    } else {
      extractSection.style.display = 'none';
    }
  } catch (error) {
    // Content Scriptが injectされていない場合などは無視
    extractSection.style.display = 'none';
  }
}

// 抽出ボタンクリック時の処理
async function handleExtract() {
  try {
    const result = await chrome.storage.local.get(['extractedNumbers']);

    if (!result.extractedNumbers || result.extractedNumbers.length === 0) {
      alert('抽出された伝票番号がありません');
      return;
    }

    // 入力フィールドにセット
    numbersInput.value = result.extractedNumbers.join('\n');

    // 抽出ボタンを非表示にする
    extractSection.style.display = 'none';
  } catch (error) {
    alert(`エラーが発生しました: ${error.message}`);
  }
}

// テスト用エクスポート（本番環境では無視される）
if (typeof exports !== 'undefined') {
  exports.extractTrackingNumbers = extractTrackingNumbers;
  exports.escapeHtml = escapeHtml;
  exports.loadLatestResults = loadLatestResults;
  exports.saveToHistory = saveToHistory;
  exports.loadHistory = loadHistory;
  exports.handleClearHistory = handleClearHistory;
  exports.checkExtractableNumbers = checkExtractableNumbers;
  exports.handleExtract = handleExtract;
  exports.handleTrack = handleTrack;
}
