/**
 * 西濃追跡チェッカー - 統合テスト
 *
 * 実際のモジュールをインポートして、複数の機能を組み合わせたテストを実行
 */

import { describe, it, expect, vi, beforeEach, afterEach } from 'vitest';

// 実際のファイルをインポート
import {
  extractTrackingNumbers,
  getStatusType,
  isValidTrackingNumber,
  escapeHtml
} from '../utils/common.js';

describe('統合テスト: 伝票番号追跡の完全なフロー', () => {
  describe('伝票番号抽出とステータス判定の統合', () => {
    it('テキストから伝票番号を抽出し、有効性をフィルタリング', () => {
      // ユーザー入力テキスト（改行区切り、カンマ区切り混在）
      const userInput = `1234567890
9876543210, 1111111111
0312345678 5555555555`;

      // 伝票番号抽出（10-12桁の数字）
      const numbers = extractTrackingNumbers(userInput);

      // extractTrackingNumbersは桁数ベースで抽出
      expect(numbers).toContain('1234567890');
      expect(numbers).toContain('9876543210');
      expect(numbers).toContain('1111111111');
      expect(numbers).toContain('5555555555');
      expect(numbers).toContain('0312345678'); // 10桁なので抽出される

      // isValidTrackingNumberで電話番号を除外
      const validNumbers = numbers.filter(n => isValidTrackingNumber(n));
      expect(validNumbers).not.toContain('0312345678'); // 電話番号除外
    });

    it('複数のステータスから正しいステータスタイプを判定する', () => {
      const testCases = [
        { status: '配達完了', expectedType: 'delivered' },
        { status: '到着しました', expectedType: 'delivered' },
        { status: 'お届け済みです', expectedType: 'delivered' },
        { status: '配達中です', expectedType: 'intransit' },
        { status: '持出中', expectedType: 'intransit' },
        { status: '発送済みです', expectedType: 'shipped' },
        { status: '出荷完了', expectedType: 'delivered' }, // 「完了」が含まれるためdelivered
        { status: '受付済みです', expectedType: 'pickedup' },
        { status: '集荷完了', expectedType: 'delivered' } // 「完了」が含まれるためdelivered
      ];

      testCases.forEach(({ status, expectedType }) => {
        const statusType = getStatusType(status);
        expect(statusType).toBe(expectedType);
      });
    });
  });

  describe('伝票番号バリデーションの統合', () => {
    it('抽出された伝票番号の有効性を検証', () => {
      const rawText = '1234567890 9876543210 5555555555';
      const numbers = extractTrackingNumbers(rawText);

      // 全ての抽出番号が有効であることを確認
      numbers.forEach(number => {
        expect(isValidTrackingNumber(number)).toBe(true);
      });
    });

    it('電話番号は無効と判定される', () => {
      // 電話番号の例
      const phoneNumbers = [
        '0312345678',  // 固定電話
        '0612345678',  // 固定電話
        '0120123456',  // フリーダイヤル
        '0570123456',  // ナビダイヤル
        '08001234567'  // フリーダイヤル
      ];

      phoneNumbers.forEach(phoneNumber => {
        expect(isValidTrackingNumber(phoneNumber)).toBe(false);
      });
    });

    it('10桁と11桁の数字は有効な伝票番号', () => {
      const validNumbers = [
        '1234567890',   // 10桁
        '12345678901',  // 11桁
        '123456789012'  // 12桁
      ];

      validNumbers.forEach(number => {
        expect(isValidTrackingNumber(number)).toBe(true);
      });
    });

    it('10桁未満または13桁超の数字は除外される', () => {
      const invalidNumbers = [
        '123456789',    // 9桁
        '1234567890123' // 13桁
      ];

      invalidNumbers.forEach(number => {
        expect(isValidTrackingNumber(number)).toBe(false);
      });
    });
  });

  describe('HTMLエスケープとセキュリティの統合', () => {
    it('伝票番号に含まれるHTML特殊文字をエスケープ', () => {
      const unsafeNumber = '<script>alert("XSS")</script>';
      const escaped = escapeHtml(unsafeNumber);

      expect(escaped).toContain('&lt;');
      expect(escaped).toContain('&gt;');
      expect(escaped).toContain('&quot;');
      expect(escaped).not.toContain('<script>');
    });

    it('ステータス情報を安全に表示するためにエスケープ', () => {
      const unsafeStatus = '<img src=x onerror=alert(1)>';
      const escaped = escapeHtml(unsafeStatus);

      // HTMLタグがエスケープされていることを確認
      expect(escaped).not.toContain('<img');
      // 属性内のテキストはエスケープされない（これは正常な動作）
      expect(escaped).toContain('&lt;img src=x onerror=alert(1)&gt;');
    });
  });
});

describe('統合テスト: 履歴保存の完全なフロー', () => {
  let storageData;
  let mockStorage;

  beforeEach(() => {
    storageData = {
      history: [],
      latestResults: null
    };

    mockStorage = {
      local: {
        get: vi.fn((keys) => {
          const result = {};
          keys.forEach(key => {
            if (storageData[key] !== undefined) {
              result[key] = storageData[key];
            }
          });
          return Promise.resolve(result);
        }),
        set: vi.fn((data) => {
          Object.assign(storageData, data);
          return Promise.resolve();
        }),
        remove: vi.fn((keys) => {
          keys.forEach(key => {
            delete storageData[key];
          });
          return Promise.resolve();
        })
      }
    };

    global.chrome = {
      storage: mockStorage
    };
  });

  afterEach(() => {
    vi.restoreAllMocks();
  });

  async function saveToHistory(results) {
    // popup.jsのsaveToHistory関数の実装を再現
    const result = await mockStorage.local.get(['history']);
    const history = result.history || [];

    // 成功したものだけを履歴に保存
    const successfulResults = results.filter(item => {
      if (item.error) return false;
      if (item.status === '不明') return false;
      return true;
    });

    if (successfulResults.length === 0) {
      return;
    }

    // 重複除去
    const newNumbers = new Set(successfulResults.map(r => r.number));
    const filteredHistory = history.filter(h => !newNumbers.has(h.number));

    // 新しい結果を先頭に追加
    const newHistory = [...successfulResults, ...filteredHistory];

    // 最大50件に制限
    const limitedHistory = newHistory.slice(0, 50);

    await mockStorage.local.set({ history: limitedHistory });
  }

  it('正常な追跡結果を履歴に保存', async () => {
    const results = [
      { number: '1234567890', status: '配達完了', statusType: 'delivered', timestamp: '2026-01-23T00:00:00Z' },
      { number: '9876543210', status: '配達中', statusType: 'intransit', timestamp: '2026-01-23T01:00:00Z' }
    ];

    await saveToHistory(results);

    expect(storageData.history).toHaveLength(2);
    expect(storageData.history[0].number).toBe('1234567890');
    expect(storageData.history[1].number).toBe('9876543210');
  });

  it('エラーありの結果は履歴に保存しない', async () => {
    const results = [
      { number: '1234567890', status: '配達完了', statusType: 'delivered', timestamp: '2026-01-23T00:00:00Z' },
      { number: '9876543210', error: 'タブが閉じられました' },
      { number: '5555555555', status: '不明', statusType: 'unknown' }
    ];

    await saveToHistory(results);

    expect(storageData.history).toHaveLength(1);
    expect(storageData.history[0].number).toBe('1234567890');
  });

  it('重複する伝票番号は新しい結果で上書き', async () => {
    storageData.history = [
      { number: '1234567890', status: '配達中', statusType: 'intransit', timestamp: '2026-01-22T00:00:00Z' }
    ];

    const results = [
      { number: '1234567890', status: '配達完了', statusType: 'delivered', timestamp: '2026-01-23T00:00:00Z' }
    ];

    await saveToHistory(results);

    expect(storageData.history).toHaveLength(1);
    expect(storageData.history[0].status).toBe('配達完了');
    expect(storageData.history[0].statusType).toBe('delivered');
  });

  it('履歴を最大50件に制限', async () => {
    // 既存履歴50件
    storageData.history = Array.from({ length: 50 }, (_, i) => ({
      number: `${1000000000 + i}`,
      status: '配達完了',
      statusType: 'delivered',
      timestamp: '2026-01-22T00:00:00Z'
    }));

    // 新規5件
    const results = Array.from({ length: 5 }, (_, i) => ({
      number: `200000000${i}`,
      status: '配達完了',
      statusType: 'delivered',
      timestamp: '2026-01-23T00:00:00Z'
    }));

    await saveToHistory(results);

    expect(storageData.history).toHaveLength(50);
    expect(storageData.history[0].number).toBe('2000000000');
    expect(storageData.history[4].number).toBe('2000000004');
  });
});

describe('統合テスト: バッチ処理と進捗管理', () => {
  it('25個の伝票番号を3つのバッチに分割して処理', () => {
    const numbers = Array.from({ length: 25 }, (_, i) => `${1234567890 + i}`);
    const MAX_PER_BATCH = 10;

    // バッチ分割ロジック
    const batches = [];
    for (let i = 0; i < numbers.length; i += MAX_PER_BATCH) {
      batches.push(numbers.slice(i, i + MAX_PER_BATCH));
    }

    expect(batches.length).toBe(3);
    expect(batches[0]).toHaveLength(10);
    expect(batches[1]).toHaveLength(10);
    expect(batches[2]).toHaveLength(5);
  });

  it('進捗表示のフォーマットを検証', () => {
    const total = 25;
    const batchCount = 3;
    const MAX_PER_BATCH = 10;

    const progressUpdates = [];
    for (let batchIndex = 0; batchIndex < batchCount; batchIndex++) {
      const completedCount = Math.min((batchIndex + 1) * MAX_PER_BATCH, total);
      const badgeText = `${completedCount}/${total}`;
      progressUpdates.push(badgeText);
    }

    expect(progressUpdates).toEqual(['10/25', '20/25', '25/25']);
  });
});

describe('統合テスト: エラーハンドリング', () => {
  it('伝票番号が見つからない場合のエラーメッセージ', () => {
    const selectionText = 'これは伝票番号を含まないテキストです';
    const numbers = extractTrackingNumbers(selectionText);

    expect(numbers).toHaveLength(0);
  });

  it('タイムアウト時のエラー結果のフォーマット', () => {
    const number = '1234567890';
    const errorResult = {
      number: number,
      status: 'エラー',
      statusType: 'error',
      dates: {},
      url: `https://track.seino.co.jp/?no=${number}`,
      error: '結果の取得がタイムアウトしました',
      timestamp: new Date().toISOString()
    };

    expect(errorResult).toHaveProperty('number');
    expect(errorResult).toHaveProperty('status', 'エラー');
    expect(errorResult).toHaveProperty('statusType', 'error');
    expect(errorResult).toHaveProperty('error');
  });

  it('タブが閉じられた場合のエラーハンドリング', () => {
    const number = '1234567890';
    const errorResult = {
      number: number,
      status: 'エラー',
      statusType: 'error',
      dates: {},
      url: `https://track.seino.co.jp/?no=${number}`,
      error: 'タブが閉じられました',
      timestamp: new Date().toISOString()
    };

    expect(errorResult.error).toBe('タブが閉じられました');
  });
});

describe('統合テスト: URL生成とページ操作', () => {
  it('西濃運輸の追跡ページURLを生成', () => {
    const trackingUrl = 'https://track.seino.co.jp/kamotsu/GempyoNoShokai.do';
    expect(trackingUrl).toBe('https://track.seino.co.jp/kamotsu/GempyoNoShokai.do');
  });

  it('伝票番号付きのURLを生成', () => {
    const number = '1234567890';
    const url = `https://track.seino.co.jp/?no=${number}`;
    expect(url).toBe('https://track.seino.co.jp/?no=1234567890');
  });
});

describe('統合テスト: 結果オブジェクトの構造', () => {
  it('成功時の結果オブジェクトを生成', () => {
    const result = {
      number: '1234567890',
      status: '配達完了',
      statusType: 'delivered',
      dates: {
        '受付': '2024-01-01 10:00',
        '配達完了': '2024-01-02 15:30'
      },
      url: 'https://track.seino.co.jp/?no=1234567890',
      timestamp: new Date().toISOString()
    };

    expect(result).toHaveProperty('number');
    expect(result).toHaveProperty('status');
    expect(result).toHaveProperty('statusType');
    expect(result).toHaveProperty('dates');
    expect(result).toHaveProperty('url');
    expect(result).toHaveProperty('timestamp');
    expect(typeof result.dates).toBe('object');
  });

  it('ステータスタイプ判定の一貫性を検証', () => {
    const statuses = [
      '配達完了', '到着', 'お届け',
      '配達中', '持出',
      '発送', '出荷',
      '受付', '集荷'
    ];

    statuses.forEach(status => {
      const statusType = getStatusType(status);
      expect(['delivered', 'intransit', 'shipped', 'pickedup']).toContain(statusType);
    });
  });
});

describe('統合テスト: 重複除外と一意性', () => {
  it('重複する伝票番号を除外', () => {
    const text = '1234567890 1234567890 9876543210 1234567890';
    const numbers = extractTrackingNumbers(text);

    expect(numbers).toHaveLength(2);
    expect(numbers).toContain('1234567890');
    expect(numbers).toContain('9876543210');
  });

  it('異なる区切り文字で同じ伝票番号を統合', () => {
    const text = '1234567890,9876543210\n1234567890';
    const numbers = extractTrackingNumbers(text);

    expect(numbers).toHaveLength(2);
  });
});

describe('統合テスト: ストレージ操作の統合', () => {
  let storageData;
  let mockStorage;

  beforeEach(() => {
    storageData = {
      trackingProgress: null,
      latestResults: null
    };

    mockStorage = {
      local: {
        get: vi.fn((keys) => {
          const result = {};
          keys.forEach(key => {
            if (storageData[key] !== undefined) {
              result[key] = storageData[key];
            }
          });
          return Promise.resolve(result);
        }),
        set: vi.fn((data) => {
          Object.assign(storageData, data);
          return Promise.resolve();
        }),
        remove: vi.fn((keys) => {
          keys.forEach(key => {
            delete storageData[key];
          });
          return Promise.resolve();
        })
      }
    };

    global.chrome = {
      storage: mockStorage
    };
  });

  afterEach(() => {
    vi.restoreAllMocks();
  });

  it('進捗を保存して復元', async () => {
    const progress = {
      total: 25,
      completed: 10,
      startTime: new Date().toISOString()
    };

    await mockStorage.local.set({ trackingProgress: progress });
    const result = await mockStorage.local.get(['trackingProgress']);

    expect(result.trackingProgress).toEqual(progress);
    expect(result.trackingProgress.total).toBe(25);
    expect(result.trackingProgress.completed).toBe(10);
  });

  it('進捗をクリア', async () => {
    storageData.trackingProgress = {
      total: 25,
      completed: 10,
      startTime: new Date().toISOString()
    };

    await mockStorage.local.remove(['trackingProgress']);
    const result = await mockStorage.local.get(['trackingProgress']);

    expect(result.trackingProgress).toBeUndefined();
  });
});

describe('統合テスト: パフォーマンスと効率性', () => {
  it('大量の伝票番号を効率的に処理', () => {
    const numbers = Array.from({ length: 100 }, (_, i) => `${1234567890 + i}`);
    const MAX_PER_BATCH = 10;

    const batches = [];
    for (let i = 0; i < numbers.length; i += MAX_PER_BATCH) {
      batches.push(numbers.slice(i, i + MAX_PER_BATCH));
    }

    expect(batches.length).toBe(10);
    batches.forEach(batch => {
      expect(batch.length).toBeLessThanOrEqual(10);
    });
  });

  it('重複除外が効率的に動作', () => {
    const duplicateNumbers = [];
    for (let i = 0; i < 100; i++) {
      duplicateNumbers.push('1234567890');
    }

    const uniqueNumbers = [...new Set(duplicateNumbers)];
    expect(uniqueNumbers).toHaveLength(1);
  });
});

describe('統合テスト: エッジケース', () => {
  it('空の配列が渡された場合', () => {
    const numbers = [];
    const MAX_PER_BATCH = 10;

    const batches = [];
    for (let i = 0; i < numbers.length; i += MAX_PER_BATCH) {
      batches.push(numbers.slice(i, i + MAX_PER_BATCH));
    }

    expect(batches.length).toBe(0);
  });

  it('1つの伝票番号のみの場合', () => {
    const numbers = ['1234567890'];
    const extracted = extractTrackingNumbers(numbers.join('\n'));

    expect(extracted).toEqual(['1234567890']);
  });

  it('正確に10個の伝票番号の場合', () => {
    const numbers = Array.from({ length: 10 }, (_, i) => `${1234567890 + i}`);
    const MAX_PER_BATCH = 10;

    const batches = [];
    for (let i = 0; i < numbers.length; i += MAX_PER_BATCH) {
      batches.push(numbers.slice(i, i + MAX_PER_BATCH));
    }

    expect(batches.length).toBe(1);
    expect(batches[0]).toHaveLength(10);
  });

  it('11個の伝票番号の場合', () => {
    const numbers = Array.from({ length: 11 }, (_, i) => `${1234567890 + i}`);
    const MAX_PER_BATCH = 10;

    const batches = [];
    for (let i = 0; i < numbers.length; i += MAX_PER_BATCH) {
      batches.push(numbers.slice(i, i + MAX_PER_BATCH));
    }

    expect(batches.length).toBe(2);
    expect(batches[0]).toHaveLength(10);
    expect(batches[1]).toHaveLength(1);
  });
});

describe('統合テスト: 定数の整合性', () => {
  it('全ての定数が期待通りに定義されている', () => {
    const DELAYS = {
      CONTENT_SCRIPT_READY: 3000,
      TAB_COMPLETE: 1500,
      CONTENT_SCRIPT_TIMEOUT: 5000,
      RESULT_TIMEOUT: 30000
    };

    const MAX_PER_BATCH = 10;
    const MAX_HISTORY_SIZE = 50;
    const STORAGE_KEY_PROGRESS = 'trackingProgress';

    expect(DELAYS.CONTENT_SCRIPT_READY).toBe(3000);
    expect(DELAYS.TAB_COMPLETE).toBe(1500);
    expect(DELAYS.CONTENT_SCRIPT_TIMEOUT).toBe(5000);
    expect(DELAYS.RESULT_TIMEOUT).toBe(30000);
    expect(MAX_PER_BATCH).toBe(10);
    expect(MAX_HISTORY_SIZE).toBe(50);
    expect(STORAGE_KEY_PROGRESS).toBe('trackingProgress');
  });
});
