/**
 * content.js ユニットテスト
 */

import { describe, it, expect, vi, beforeEach, afterEach } from 'vitest';
import { isValidTrackingNumber, getStatusType } from '../utils/common.js';

// content.jsで使用されている関数をモック化せずにテストするため、
// DOMとChrome APIをモック化してからインポート
const mockChrome = {
  runtime: {
    sendMessage: vi.fn(),
    onMessage: {
      addListener: vi.fn()
    }
  }
};

// モックをグローバルに設定
global.chrome = mockChrome;

// URLモック
const mockUrl = {
  search: '',
  hash: ''
};

// URLSearchParamsモック
class MockURLSearchParams {
  constructor(search) {
    this.params = new Map();
    if (search) {
      search.substring(1).split('&').forEach(param => {
        const [key, value] = param.split('=');
        if (key) {
          this.params.set(key, value || '');
        }
      });
    }
  }

  has(name) {
    return this.params.has(name);
  }

  get(name) {
    return this.params.get(name);
  }
}

// content.jsをインポート
// 注意: ESモジュールなので動的インポートが必要
// ここではテスト用に実装を再現する

// テスト用にcontent.jsの関数を再定義（テスト用）
const INPUT_FIELDS = ['GNPNO1', 'GNPNO2', 'GNPNO3', 'GNPNO4', 'GNPNO5',
                      'GNPNO6', 'GNPNO7', 'GNPNO8', 'GNPNO9', 'GNPNOA'];

// URLモックのセットアップ関数
function setupUrlMock(urlParams = {}) {
  const params = new URLSearchParams();
  Object.entries(urlParams).forEach(([key, value]) => {
    params.set(key, value);
  });
  mockUrl.search = params.toString();
  return mockUrl;
}

// DOMモックのセットアップ関数
function setupDomMock(config = {}) {
  const mockInputs = {};
  const mockTables = [];

  // 入力フィールドモックの作成
  INPUT_FIELDS.forEach(fieldName => {
    const mockInput = {
      name: fieldName,
      value: config.inputValues?.[fieldName] || '',
      get trimmedValue() {
        return this.value?.trim() || '';
      }
    };
    mockInputs[fieldName] = mockInput;
  });

  // querySelectorのモック
  global.document = {
    querySelector: vi.fn((selector) => {
      // input[name="xxx"]のパターン
      const inputMatch = selector.match(/input\[name="([^"]+)"\]/);
      if (inputMatch) {
        const fieldName = inputMatch[1];
        return mockInputs[fieldName] || null;
      }
      // その他のセレクタ
      return config.querySelectorResult || null;
    }),
    querySelectorAll: vi.fn((selector) => {
      if (selector === 'table') {
        return mockTables;
      }
      return [];
    }),
    body: {
      textContent: config.bodyText || ''
    },
    createElement: vi.fn((tagName) => ({
      tagName,
      style: {}
    }))
  };

  // テーブルモックの作成
  if (config.tables) {
    config.tables.forEach(tableConfig => {
      const mockTable = {
        querySelectorAll: vi.fn((selector) => {
          if (selector === 'tr') {
            return tableConfig.rows || [];
          }
          if (selector === 'th') {
            return tableConfig.headers || [];
          }
          return [];
        })
      };
      mockTables.push(mockTable);
    });
  }

  return { mockInputs, mockTables };
}

// MutationObserverモック
function setupMutationObserverMock() {
  let callback = null;
  const mockObserver = {
    observe: vi.fn(),
    disconnect: vi.fn(),
    _trigger: () => {
      if (callback) callback();
    }
  };

  global.MutationObserver = vi.fn().mockImplementation((cb) => {
    callback = cb;
    return mockObserver;
  });

  return mockObserver;
}

// ============================================================================
// checkIsResultPage のテスト
// ============================================================================

describe('checkIsResultPage', () => {
  let originalWindow;
  let originalURLSearchParams;

  beforeEach(() => {
    originalWindow = global.window;
    originalURLSearchParams = global.URLSearchParams;
  });

  afterEach(() => {
    global.window = originalWindow;
    global.URLSearchParams = originalURLSearchParams;
  });

  it('URLパラメータにGNPNO1がある場合はtrueを返す', () => {
    // URLSearchParamsのモック
    class MockURLSearchParams {
      constructor(search) {
        this.params = new Map();
        if (search && search.includes('GNPNO1')) {
          this.params.set('GNPNO1', '1234567890');
        }
      }

      has(name) {
        return this.params.has(name);
      }
    }

    global.URLSearchParams = MockURLSearchParams;

    global.window = {
      location: { search: '?GNPNO1=1234567890' }
    };

    setupDomMock();

    // 実装の再現
    function checkIsResultPage() {
      const urlParams = new URLSearchParams(window.location.search);
      const hasGnpNo = urlParams.has('GNPNO1') || urlParams.has('GNPNO2') ||
                        urlParams.has('GNPNO3') || urlParams.has('GNPNO4') ||
                        urlParams.has('GNPNO5') || urlParams.has('GNPNO6') ||
                        urlParams.has('GNPNO7') || urlParams.has('GNPNO8') ||
                        urlParams.has('GNPNO9') || urlParams.has('GNPNOA');

      if (hasGnpNo) {
        return true;
      }
      return false;
    }

    expect(checkIsResultPage()).toBe(true);
  });

  it('URLパラメータがない場合は入力フィールドをチェック', () => {
    class MockURLSearchParams {
      constructor() {
        this.params = new Map();
      }

      has(name) {
        return this.params.has(name);
      }
    }

    global.URLSearchParams = MockURLSearchParams;

    global.window = {
      location: { search: '' }
    };

    setupDomMock({
      inputValues: {
        GNPNO1: '1234567890',
        GNPNO2: ''
      }
    });

    // 実装の再現
    function checkIsResultPage() {
      const urlParams = new URLSearchParams(window.location.search);
      const hasGnpNo = urlParams.has('GNPNO1') || urlParams.has('GNPNO2') ||
                        urlParams.has('GNPNO3') || urlParams.has('GNPNO4') ||
                        urlParams.has('GNPNO5') || urlParams.has('GNPNO6') ||
                        urlParams.has('GNPNO7') || urlParams.has('GNPNO8') ||
                        urlParams.has('GNPNO9') || urlParams.has('GNPNOA');

      if (hasGnpNo) {
        return true;
      }

      let hasInputValues = false;
      INPUT_FIELDS.forEach(fieldName => {
        const input = document.querySelector(`input[name="${fieldName}"]`);
        if (input && input.value && input.value.trim()) {
          hasInputValues = true;
        }
      });

      if (hasInputValues) {
        return true;
      }
      return false;
    }

    expect(checkIsResultPage()).toBe(true);
  });

  it('全ての条件に該当しない場合はfalseを返す', () => {
    class MockURLSearchParams {
      constructor() {
        this.params = new Map();
      }

      has(name) {
        return this.params.has(name);
      }
    }

    global.URLSearchParams = MockURLSearchParams;

    global.window = {
      location: { search: '' }
    };

    setupDomMock({
      inputValues: {}
    });

    function checkIsResultPage() {
      const urlParams = new URLSearchParams(window.location.search);
      const hasGnpNo = urlParams.has('GNPNO1') || urlParams.has('GNPNO2') ||
                        urlParams.has('GNPNO3') || urlParams.has('GNPNO4') ||
                        urlParams.has('GNPNO5') || urlParams.has('GNPNO6') ||
                        urlParams.has('GNPNO7') || urlParams.has('GNPNO8') ||
                        urlParams.has('GNPNO9') || urlParams.has('GNPNOA');

      if (hasGnpNo) {
        return true;
      }

      let hasInputValues = false;
      INPUT_FIELDS.forEach(fieldName => {
        const input = document.querySelector(`input[name="${fieldName}"]`);
        if (input && input.value && input.value.trim()) {
          hasInputValues = true;
        }
      });

      if (hasInputValues) {
        return true;
      }
      return false;
    }

    expect(checkIsResultPage()).toBe(false);
  });
});

// ============================================================================
// extractNumbersFromPage のテスト
// ============================================================================

describe('extractNumbersFromPage', () => {
  beforeEach(() => {
    setupDomMock();
  });

  it('テーブルから伝票番号を抽出する', () => {
    const mockCells = [
      { textContent: '1234567890' }
    ];

    const mockRow = {
      textContent: '1234567890',
      querySelectorAll: vi.fn(() => mockCells)
    };

    const mockHeaders = [
      { textContent: '伝票ナンバー' },
      { textContent: 'ステータス' }
    ];

    const mockTable = {
      querySelectorAll: vi.fn((selector) => {
        if (selector === 'th') {
          return mockHeaders;
        }
        if (selector === 'tbody tr') {
          return [mockRow];
        }
        return [];
      })
    };

    global.document.querySelectorAll = vi.fn((selector) => {
      if (selector === 'table') {
        return [mockTable];
      }
      return [];
    });

    // 実装の再現
    function extractNumbersFromPage() {
      const numbers = [];
      const tables = document.querySelectorAll('table');

      tables.forEach(table => {
        const headers = table.querySelectorAll('th');
        let targetColumnIndex = -1;

        headers.forEach((th, index) => {
          const text = th.textContent.trim();
          if (text.includes('伝票ナンバー') || text.includes('伝票番号') ||
              text.includes('伝票No') || text.includes('送り状番号')) {
            targetColumnIndex = index;
          }
        });

        if (targetColumnIndex >= 0) {
          const rows = table.querySelectorAll('tbody tr');

          rows.forEach(row => {
            const cells = row.querySelectorAll('td');
            if (cells.length > targetColumnIndex) {
              const cellText = cells[targetColumnIndex].textContent.trim();
              const matches = cellText.match(/\d{10,12}/g);
              if (matches) {
                matches.forEach(m => {
                  if (isValidTrackingNumber(m)) {
                    numbers.push(m);
                  }
                });
              }
            }
          });
        }
      });

      return [...new Set(numbers)];
    }

    const result = extractNumbersFromPage();
    expect(result).toEqual(['1234567890']);
  });

  it('電話番号は除外する', () => {
    // 電話番号と伝票番号が混在するセル
    const mockCell = {
      textContent: '0312345678 1234567890'
    };

    const mockRow = {
      textContent: '0312345678 1234567890',
      querySelectorAll: vi.fn(() => [mockCell])
    };

    const mockHeaders = [{ textContent: '伝票ナンバー' }];

    const mockTable = {
      querySelectorAll: vi.fn((selector) => {
        if (selector === 'th') {
          return mockHeaders;
        }
        if (selector === 'tbody tr') {
          return [mockRow];
        }
        return [];
      })
    };

    global.document.querySelectorAll = vi.fn((selector) => {
      if (selector === 'table') {
        return [mockTable];
      }
      return [];
    });

    function extractNumbersFromPage() {
      const numbers = [];
      const tables = document.querySelectorAll('table');

      tables.forEach(table => {
        const headers = table.querySelectorAll('th');
        let targetColumnIndex = -1;

        headers.forEach((th, index) => {
          const text = th.textContent.trim();
          if (text.includes('伝票ナンバー') || text.includes('伝票番号') ||
              text.includes('伝票No') || text.includes('送り状番号')) {
            targetColumnIndex = index;
          }
        });

        if (targetColumnIndex >= 0) {
          const rows = table.querySelectorAll('tbody tr');

          rows.forEach(row => {
            const cells = row.querySelectorAll('td');
            if (cells.length > targetColumnIndex) {
              const cellText = cells[targetColumnIndex].textContent.trim();
              const matches = cellText.match(/\d{10,12}/g);
              if (matches) {
                matches.forEach(m => {
                  if (isValidTrackingNumber(m)) {
                    numbers.push(m);
                  }
                });
              }
            }
          });
        }
      });

      return [...new Set(numbers)];
    }

    const result = extractNumbersFromPage();
    expect(result).toEqual(['1234567890']);
    expect(result).not.toContain('0312345678');
  });

  it('重複を除外する', () => {
    const mockCells1 = [{ textContent: '1234567890' }];
    const mockCells2 = [{ textContent: '1234567890' }];

    const mockRow1 = {
      textContent: '1234567890',
      querySelectorAll: vi.fn(() => mockCells1)
    };

    const mockRow2 = {
      textContent: '1234567890',
      querySelectorAll: vi.fn(() => mockCells2)
    };

    const mockHeaders = [{ textContent: '伝票ナンバー' }];

    const mockTable = {
      querySelectorAll: vi.fn((selector) => {
        if (selector === 'th') {
          return mockHeaders;
        }
        if (selector === 'tbody tr') {
          return [mockRow1, mockRow2];
        }
        return [];
      })
    };

    global.document.querySelectorAll = vi.fn((selector) => {
      if (selector === 'table') {
        return [mockTable];
      }
      return [];
    });

    function extractNumbersFromPage() {
      const numbers = [];
      const tables = document.querySelectorAll('table');

      tables.forEach(table => {
        const headers = table.querySelectorAll('th');
        let targetColumnIndex = -1;

        headers.forEach((th, index) => {
          const text = th.textContent.trim();
          if (text.includes('伝票ナンバー') || text.includes('伝票番号') ||
              text.includes('伝票No') || text.includes('送り状番号')) {
            targetColumnIndex = index;
          }
        });

        if (targetColumnIndex >= 0) {
          const rows = table.querySelectorAll('tbody tr');

          rows.forEach(row => {
            const cells = row.querySelectorAll('td');
            if (cells.length > targetColumnIndex) {
              const cellText = cells[targetColumnIndex].textContent.trim();
              const matches = cellText.match(/\d{10,12}/g);
              if (matches) {
                matches.forEach(m => {
                  if (isValidTrackingNumber(m)) {
                    numbers.push(m);
                  }
                });
              }
            }
          });
        }
      });

      return [...new Set(numbers)];
    }

    const result = extractNumbersFromPage();
    expect(result).toEqual(['1234567890']);
  });

  it('テーブルがない場合はフォールバックして本文から抽出', () => {
    setupDomMock({
      tables: [],
      bodyText: '伝票番号: 1234567890, 9876543210'
    });

    function extractNumbersFromPage() {
      const numbers = [];
      const tables = document.querySelectorAll('table');

      tables.forEach(table => {
        const headers = table.querySelectorAll('th');
        let targetColumnIndex = -1;

        headers.forEach((th, index) => {
          const text = th.textContent.trim();
          if (text.includes('伝票ナンバー') || text.includes('伝票番号') ||
              text.includes('伝票No') || text.includes('送り状番号')) {
            targetColumnIndex = index;
          }
        });

        if (targetColumnIndex >= 0) {
          const rows = table.querySelectorAll('tbody tr');

          rows.forEach(row => {
            const cells = row.querySelectorAll('td');
            if (cells.length > targetColumnIndex) {
              const cellText = cells[targetColumnIndex].textContent.trim();
              const matches = cellText.match(/\d{10,12}/g);
              if (matches) {
                matches.forEach(m => {
                  if (isValidTrackingNumber(m)) {
                    numbers.push(m);
                  }
                });
              }
            }
          });
        }
      });

      if (numbers.length === 0) {
        const bodyText = document.body.textContent;
        const matches = bodyText.match(/\b\d{10,12}\b/g);
        if (matches) {
          matches.forEach(m => {
            if (isValidTrackingNumber(m)) {
              numbers.push(m);
            }
          });
        }
      }

      return [...new Set(numbers)];
    }

    const result = extractNumbersFromPage();
    expect(result).toEqual(['1234567890', '9876543210']);
  });
});

// ============================================================================
// trackNumbersAndSubmit のテスト
// ============================================================================

describe('trackNumbersAndSubmit', () => {
  beforeEach(() => {
    setupDomMock();
  });

  it('伝票番号を入力フィールドにセットする', () => {
    const mockSearchButton = {
      click: vi.fn()
    };

    const { mockInputs } = setupDomMock({
      querySelectorResult: mockSearchButton,
      inputValues: {}
    });

    // querySelectorモックを更新
    global.document.querySelector = vi.fn((selector) => {
      if (selector.startsWith('input[name="')) {
        const fieldName = selector.match(/input\[name="([^"]+)"\]/)[1];
        return mockInputs[fieldName];
      }
      if (selector === 'input[type="submit"], input[type="image"], button[type="submit"]') {
        return mockSearchButton;
      }
      return null;
    });

    function trackNumbersAndSubmit(numbers) {
      INPUT_FIELDS.forEach(fieldName => {
        const input = document.querySelector(`input[name="${fieldName}"]`);
        if (input) {
          input.value = '';
        }
      });

      numbers.forEach((number, index) => {
        if (index < INPUT_FIELDS.length) {
          const fieldName = INPUT_FIELDS[index];
          const input = document.querySelector(`input[name="${fieldName}"]`);
          if (input) {
            input.value = number;
          }
        }
      });

      const searchButton = document.querySelector('input[type="submit"], input[type="image"], button[type="submit"]');
      if (searchButton) {
        searchButton.click();
      }
    }

    trackNumbersAndSubmit(['1234567890', '9876543210']);

    expect(mockInputs['GNPNO1'].value).toBe('1234567890');
    expect(mockInputs['GNPNO2'].value).toBe('9876543210');
    expect(mockInputs['GNPNO3'].value).toBe('');
    expect(mockSearchButton.click).toHaveBeenCalled();
  });

  it('入力フィールドをクリアしてからセットする', () => {
    const mockSearchButton = {
      click: vi.fn()
    };

    const { mockInputs } = setupDomMock({
      querySelectorResult: mockSearchButton,
      inputValues: {
        GNPNO1: '0000000000',
        GNPNO2: '1111111111'
      }
    });

    global.document.querySelector = vi.fn((selector) => {
      if (selector.startsWith('input[name="')) {
        const fieldName = selector.match(/input\[name="([^"]+)"\]/)[1];
        return mockInputs[fieldName];
      }
      if (selector === 'input[type="submit"], input[type="image"], button[type="submit"]') {
        return mockSearchButton;
      }
      return null;
    });

    function trackNumbersAndSubmit(numbers) {
      INPUT_FIELDS.forEach(fieldName => {
        const input = document.querySelector(`input[name="${fieldName}"]`);
        if (input) {
          input.value = '';
        }
      });

      numbers.forEach((number, index) => {
        if (index < INPUT_FIELDS.length) {
          const fieldName = INPUT_FIELDS[index];
          const input = document.querySelector(`input[name="${fieldName}"]`);
          if (input) {
            input.value = number;
          }
        }
      });
    }

    trackNumbersAndSubmit(['1234567890']);

    expect(mockInputs['GNPNO1'].value).toBe('1234567890');
    expect(mockInputs['GNPNO2'].value).toBe('');
  });

  it('検索ボタンがない場合はフォームを送信', () => {
    const mockForm = {
      submit: vi.fn()
    };

    const { mockInputs } = setupDomMock({
      inputValues: {}
    });

    global.document.querySelector = vi.fn((selector) => {
      if (selector.startsWith('input[name="')) {
        const fieldName = selector.match(/input\[name="([^"]+)"\]/)[1];
        return mockInputs[fieldName];
      }
      if (selector === 'input[type="submit"], input[type="image"], button[type="submit"]') {
        return null;
      }
      if (selector === 'form') {
        return mockForm;
      }
      return null;
    });

    function trackNumbersAndSubmit(numbers) {
      INPUT_FIELDS.forEach(fieldName => {
        const input = document.querySelector(`input[name="${fieldName}"]`);
        if (input) {
          input.value = '';
        }
      });

      numbers.forEach((number, index) => {
        if (index < INPUT_FIELDS.length) {
          const fieldName = INPUT_FIELDS[index];
          const input = document.querySelector(`input[name="${fieldName}"]`);
          if (input) {
            input.value = number;
          }
        }
      });

      const searchButton = document.querySelector('input[type="submit"], input[type="image"], button[type="submit"]');

      if (searchButton) {
        searchButton.click();
      } else {
        const form = document.querySelector('form');
        if (form) {
          form.submit();
        } else {
          throw new Error('検索ボタンとフォームが見つかりません');
        }
      }
    }

    expect(() => trackNumbersAndSubmit(['1234567890'])).not.toThrow();
    expect(mockForm.submit).toHaveBeenCalled();
  });

  it('検索ボタンもフォームもない場合はエラー', () => {
    const { mockInputs } = setupDomMock({
      inputValues: {}
    });

    global.document.querySelector = vi.fn((selector) => {
      if (selector.startsWith('input[name="')) {
        const fieldName = selector.match(/input\[name="([^"]+)"\]/)[1];
        return mockInputs[fieldName];
      }
      return null;
    });

    function trackNumbersAndSubmit(numbers) {
      INPUT_FIELDS.forEach(fieldName => {
        const input = document.querySelector(`input[name="${fieldName}"]`);
        if (input) {
          input.value = '';
        }
      });

      numbers.forEach((number, index) => {
        if (index < INPUT_FIELDS.length) {
          const fieldName = INPUT_FIELDS[index];
          const input = document.querySelector(`input[name="${fieldName}"]`);
          if (input) {
            input.value = number;
          }
        }
      });

      const searchButton = document.querySelector('input[type="submit"], input[type="image"], button[type="submit"]');

      if (searchButton) {
        searchButton.click();
      } else {
        const form = document.querySelector('form');
        if (form) {
          form.submit();
        } else {
          throw new Error('検索ボタンとフォームが見つかりません');
        }
      }
    }

    expect(() => trackNumbersAndSubmit(['1234567890'])).toThrow('検索ボタンとフォームが見つかりません');
  });
});

// ============================================================================
// extractAndSendAllStatuses のテスト
// ============================================================================

describe('extractAndSendAllStatuses', () => {
  beforeEach(() => {
    setupDomMock();
    mockChrome.runtime.sendMessage.mockClear();
  });

  it('テーブルからステータスを抽出して送信', () => {
    const mockRow = {
      textContent: '1234567890 配達済みです',
      querySelectorAll: vi.fn(() => [
        { textContent: '1234567890' },
        { textContent: '配達済みです' }
      ])
    };

    const mockTable = {
      querySelectorAll: vi.fn((selector) => {
        if (selector === 'tr') {
          return [mockRow];
        }
        return [];
      })
    };

    setupDomMock({
      tables: [{
        rows: [mockRow]
      }],
      inputValues: {}
    });

    global.window = {
      location: { href: 'https://example.com' }
    };

    function extractAndSendAllStatuses() {
      const results = [];
      const tables = document.querySelectorAll('table');

      tables.forEach((table, tableIndex) => {
        const rows = table.querySelectorAll('tr');

        rows.forEach((row, rowIndex) => {
          const rowText = row.textContent;

          const numberMatch = rowText.match(/\b(\d{10,12})\b/);

          if (numberMatch && isValidTrackingNumber(numberMatch[1])) {
            const number = numberMatch[1];
            let status = '不明';
            let statusType = 'unknown';

            if (rowText.includes('配達済み')) {
              status = '配達済みです';
              statusType = 'delivered';
            }

            if (!results.find(r => r.number === number)) {
              results.push({
                number: number,
                status: status,
                statusType: statusType,
                dates: {},
                url: window.location.href,
                timestamp: new Date().toISOString()
              });
            }
          }
        });
      });

      if (results.length > 0) {
        chrome.runtime.sendMessage({
          action: 'trackingResults',
          data: results
        });
      }
    }

    extractAndSendAllStatuses();

    expect(mockChrome.runtime.sendMessage).toHaveBeenCalledWith({
      action: 'trackingResults',
      data: expect.arrayContaining([
        expect.objectContaining({
          number: '1234567890',
          status: '配達済みです',
          statusType: 'delivered'
        })
      ])
    });
  });

  it('エラーメッセージを抽出', () => {
    const mockRow = {
      textContent: '1234567890 入力されたお問合せ番号が見当りません',
      querySelectorAll: vi.fn(() => [
        { textContent: '1234567890' },
        { textContent: '見当りません' }
      ])
    };

    setupDomMock({
      tables: [{
        rows: [mockRow]
      }]
    });

    global.window = {
      location: { href: 'https://example.com' }
    };

    function extractAndSendAllStatuses() {
      const results = [];
      const tables = document.querySelectorAll('table');

      tables.forEach((table, tableIndex) => {
        const rows = table.querySelectorAll('tr');

        rows.forEach((row, rowIndex) => {
          const rowText = row.textContent;

          const numberMatch = rowText.match(/\b(\d{10,12})\b/);

          if (numberMatch && isValidTrackingNumber(numberMatch[1])) {
            const number = numberMatch[1];
            let status = '不明';
            let statusType = 'unknown';

            if (rowText.includes('見当りません') || rowText.includes('見当たりません')) {
              status = '入力されたお問合せ番号が見当りません';
              statusType = 'error';
            }

            if (!results.find(r => r.number === number)) {
              results.push({
                number: number,
                status: status,
                statusType: statusType,
                dates: {},
                url: window.location.href,
                timestamp: new Date().toISOString()
              });
            }
          }
        });
      });

      if (results.length > 0) {
        chrome.runtime.sendMessage({
          action: 'trackingResults',
          data: results
        });
      }
    }

    extractAndSendAllStatuses();

    expect(mockChrome.runtime.sendMessage).toHaveBeenCalledWith({
      action: 'trackingResults',
      data: expect.arrayContaining([
        expect.objectContaining({
          number: '1234567890',
          status: '入力されたお問合せ番号が見当りません',
          statusType: 'error'
        })
      ])
    });
  });

  it('結果が0件の場合は空配列を送信', () => {
    setupDomMock({
      tables: [{}],
      inputValues: {}
    });

    function extractAndSendAllStatuses() {
      const results = [];
      const tables = document.querySelectorAll('table');

      tables.forEach((table, tableIndex) => {
        const rows = table.querySelectorAll('tr');

        rows.forEach((row, rowIndex) => {
          const rowText = row.textContent;
          const numberMatch = rowText.match(/\b(\d{10,12})\b/);

          if (numberMatch && isValidTrackingNumber(numberMatch[1])) {
            // 処理...
          }
        });
      });

      if (results.length > 0) {
        chrome.runtime.sendMessage({
          action: 'trackingResults',
          data: results
        });
      } else {
        chrome.runtime.sendMessage({
          action: 'trackingResults',
          data: []
        });
      }
    }

    extractAndSendAllStatuses();

    expect(mockChrome.runtime.sendMessage).toHaveBeenCalledWith({
      action: 'trackingResults',
      data: []
    });
  });
});

// ============================================================================
// MutationObserverのテスト
// ============================================================================

describe('MutationObserver', () => {
  it('DOM変化を検出してコールバックを実行', () => {
    let capturedCallback = null;

    // クラスとしてモックを実装
    class MockMutationObserver {
      constructor(callback) {
        capturedCallback = callback;
      }

      observe = vi.fn();
      disconnect = vi.fn();
    }

    global.MutationObserver = MockMutationObserver;

    // MutationObserverがコンストラクタとして呼ばれた
    const testCallback = () => {};

    const observer = new global.MutationObserver(testCallback);
    expect(typeof capturedCallback).toBe('function');

    // コールバックを実行
    if (capturedCallback) {
      capturedCallback();
    }

    // observer.observeが呼ばれていないことを確認（まだセットアップのみ）
    expect(observer.observe).not.toHaveBeenCalled();
  });

  it('observeメソッドが呼ばれる', () => {
    const mockBody = {};
    const observer = setupMutationObserverMock();

    observer.observe(mockBody, {
      childList: true,
      subtree: true
    });

    expect(observer.observe).toHaveBeenCalledWith(mockBody, {
      childList: true,
      subtree: true
    });
  });

  it('disconnectメソッドが呼ばれる', () => {
    const observer = setupMutationObserverMock();

    observer.disconnect();

    expect(observer.disconnect).toHaveBeenCalled();
  });
});

// ============================================================================
// Chrome API メッセージ送信のテスト
// ============================================================================

describe('Chrome API Integration', () => {
  beforeEach(() => {
    mockChrome.runtime.sendMessage.mockClear();
    mockChrome.runtime.onMessage.addListener.mockClear();
  });

  it('contentScriptReadyメッセージを送信', () => {
    chrome.runtime.sendMessage({
      action: 'contentScriptReady',
    });

    expect(mockChrome.runtime.sendMessage).toHaveBeenCalledWith({
      action: 'contentScriptReady',
    });
  });

  it('trackNumbersメッセージを受信して処理', () => {
    let messageHandler = null;

    mockChrome.runtime.onMessage.addListener.mockImplementation((callback) => {
      messageHandler = callback;
    });

    // メッセージリスナーを登録（content.jsの初期化処理）
    chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
      if (request.action === 'trackNumbers') {
        sendResponse({ success: true, message: '2件の検索を実行中' });
      }
      return false;
    });

    // モックが呼ばれたことを確認
    expect(mockChrome.runtime.onMessage.addListener).toHaveBeenCalled();

    // メッセージハンドラーが存在することを確認
    expect(messageHandler).toBeTruthy();
  });

  it('extractNumbersFromPageメッセージを受信して処理', () => {
    let messageHandler = null;

    mockChrome.runtime.onMessage.addListener.mockImplementation((callback) => {
      messageHandler = callback;
    });

    chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
      if (request.action === 'extractNumbersFromPage') {
        sendResponse({ success: true, numbers: ['1234567890'] });
      }
      return false;
    });

    expect(mockChrome.runtime.onMessage.addListener).toHaveBeenCalled();
    expect(messageHandler).toBeTruthy();
  });

  it('sendMessageのエラーをハンドル', () => {
    mockChrome.runtime.sendMessage.mockRejectedValueOnce(new Error('Connection error'));

    chrome.runtime.sendMessage({
      action: 'contentScriptReady',
    }).catch(err => {
      expect(err.message).toBe('Connection error');
    });

    expect(mockChrome.runtime.sendMessage).toHaveBeenCalledWith({
      action: 'contentScriptReady',
    });
  });
});

// ============================================================================
// ステータス判定のテスト
// ============================================================================

describe('Status Type Detection', () => {
  it('配達済みステータスを正しく判定', () => {
    expect(getStatusType('配達完了')).toBe('delivered');
    expect(getStatusType('到着完了')).toBe('delivered');
    expect(getStatusType('お届け完了')).toBe('delivered');
  });

  it('配達中ステータスを正しく判定', () => {
    expect(getStatusType('配達中')).toBe('intransit');
    expect(getStatusType('持出')).toBe('intransit');
    expect(getStatusType('配達持出')).toBe('intransit');
  });

  it('発送ステータスを正しく判定', () => {
    expect(getStatusType('発送済み')).toBe('shipped');
    expect(getStatusType('出荷')).toBe('shipped');
  });

  it('受付ステータスを正しく判定', () => {
    expect(getStatusType('受付済み')).toBe('pickedup');
    expect(getStatusType('集荷済み')).toBe('pickedup');
  });

  it('不明なステータスを正しく判定', () => {
    expect(getStatusType('不明')).toBe('unknown');
    expect(getStatusType('')).toBe('unknown');
    expect(getStatusType(undefined)).toBe('unknown');
    expect(getStatusType(null)).toBe('unknown');
  });
});
