/**
 * 西濃追跡チェッカー - Content Script
 *
 * 機能:
 * - 西濃運輸の追跡ページで複数の伝票番号を入力して送信
 * - 結果ページで全てのステータスを抽出してBackgroundに送信
 *
 * 注意: common.jsはmanifest.jsonで先に読み込まれているため、
 * ここではimport文を使用せず、グローバルスコープの関数を直接使用します。
 */

// 入力フィールドの名前（GNPNO1〜GNPNO9、GNPNOA）
const INPUT_FIELDS = ['GNPNO1', 'GNPNO2', 'GNPNO3', 'GNPNO4', 'GNPNO5',
                      'GNPNO6', 'GNPNO7', 'GNPNO8', 'GNPNO9', 'GNPNOA'];

// 定数
const DELAYS = {
  RESULT_PAGE_OBSERVE_TIMEOUT: 25000,
  EXTRACTION_DELAY: 2000,
  PAGE_LOAD_DELAY: 2000
};

// URLかどうかを判定（結果ページかどうか）
function checkIsResultPage() {
  // 方法1: URLパラメータチェック
  const urlParams = new URLSearchParams(window.location.search);
  const hasGnpNo = urlParams.has('GNPNO1') || urlParams.has('GNPNO2') ||
                    urlParams.has('GNPNO3') || urlParams.has('GNPNO4') ||
                    urlParams.has('GNPNO5') || urlParams.has('GNPNO6') ||
                    urlParams.has('GNPNO7') || urlParams.has('GNPNO8') ||
                    urlParams.has('GNPNO9') || urlParams.has('GNPNOA');

  if (hasGnpNo) {
    return true;
  }

  // 方法2: 入力フィールドに値が入っているかチェック
  let hasInputValues = false;
  INPUT_FIELDS.forEach(fieldName => {
    const input = document.querySelector(`input[name="${fieldName}"]`);
    if (input && input.value && input.value.trim()) {
      hasInputValues = true;
    }
  });

  if (hasInputValues) {
    return true;
  }

  // 方法3: 結果テーブルが存在するかチェック
  const tables = document.querySelectorAll('table');
  for (const table of tables) {
    const rows = table.querySelectorAll('tr');
    for (const row of rows) {
      const cells = row.querySelectorAll('td');
      if (cells.length > 0) {
        const firstCell = cells[0].textContent.trim();
        // 10-12桁の数字があれば結果ページ
        if (/^\d{10,12}$/.test(firstCell)) {
          return true;
        }
      }
    }
  }

  return false;
}

// Content Scriptの準備完了をBackgroundに通知
chrome.runtime.sendMessage({
  action: 'contentScriptReady',
}).catch(err => {
  // Backgroundが待っていない場合は無視
});

// Backgroundからのメッセージを受信
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'trackNumbers') {
    // 複数の伝票番号を一度に入力して送信
    try {
      trackNumbersAndSubmit(request.numbers);
      sendResponse({ success: true, message: `${request.numbers.length}件の検索を実行中` });
    } catch (error) {
      sendResponse({ success: false, error: error.message });
    }
    return false; // 同期レスポンス
  }

  if (request.action === 'extractNumbersFromPage') {
    // ページから伝票番号を抽出
    try {
      const numbers = extractNumbersFromPage();
      sendResponse({ success: true, numbers: numbers });
    } catch (error) {
      sendResponse({ success: false, error: error.message, numbers: [] });
    }
    return false; // 同期レスポンス
  }
});

// ページから伝票番号を抽出
function extractNumbersFromPage() {
  const numbers = [];

  // 方法1: テーブル内の「伝票ナンバー」または「伝票番号」列を探す
  const tables = document.querySelectorAll('table');

  tables.forEach(table => {
    const headers = table.querySelectorAll('th');
    let targetColumnIndex = -1;

    // ヘッダーから「伝票ナンバー」または「伝票番号」列を探す
    headers.forEach((th, index) => {
      const text = th.textContent.trim();
      if (text.includes('伝票ナンバー') || text.includes('伝票番号') || text.includes('伝票No') || text.includes('送り状番号')) {
        targetColumnIndex = index;
      }
    });

    // 見つかった場合はその列のデータを抽出
    if (targetColumnIndex >= 0) {
      const rows = table.querySelectorAll('tbody tr');

      rows.forEach(row => {
        const cells = row.querySelectorAll('td');
        if (cells.length > targetColumnIndex) {
          const cellText = cells[targetColumnIndex].textContent.trim();
          // 10-12桁の数字を抽出
          const matches = cellText.match(/\d{10,12}/g);
          if (matches) {
            matches.forEach(m => {
              if (isValidTrackingNumber(m)) {
                numbers.push(m);
              }
            });
          }
        }
      });
    }
  });

  // 方法2: ページ全体から伝票番号っぽい数字を抽出（フォールバック）
  if (numbers.length === 0) {
    const bodyText = document.body.textContent;
    const matches = bodyText.match(/\b\d{10,12}\b/g);
    if (matches) {
      matches.forEach(m => {
        if (isValidTrackingNumber(m)) {
          numbers.push(m);
        }
      });
    }
  }

  // 重複を削除して返す
  const uniqueNumbers = [...new Set(numbers)];

  return uniqueNumbers;
}

// 複数の伝票番号を入力して送信（即座に返す）
function trackNumbersAndSubmit(numbers) {
  // 全ての入力フィールドをクリア
  INPUT_FIELDS.forEach(fieldName => {
    const input = document.querySelector(`input[name="${fieldName}"]`);
    if (input) {
      input.value = '';
    }
  });

  // 伝票番号を入力フィールドにセット
  numbers.forEach((number, index) => {
    if (index < INPUT_FIELDS.length) {
      const fieldName = INPUT_FIELDS[index];
      const input = document.querySelector(`input[name="${fieldName}"]`);
      if (input) {
        input.value = number;
      }
    }
  });

  // 検索ボタンを探してクリック
  const searchButton = document.querySelector('input[type="submit"], input[type="image"], button[type="submit"]');

  if (searchButton) {
    searchButton.click();

    // DOM変化を監視して結果ページを検出
    observeResultPage();
  } else {
    // ボタンが見つからない場合はフォームを送信
    const form = document.querySelector('form');
    if (form) {
      form.submit();

      // DOM変化を監視して結果ページを検出
      observeResultPage();
    } else {
      throw new Error('検索ボタンとフォームが見つかりません');
    }
  }
}

// DOM変化を監視して結果ページを検出
function observeResultPage() {
  // 重複実行を防ぐフラグ
  let hasExtracted = false;

  const observer = new MutationObserver(() => {
    if (hasExtracted) return; // 既に抽出済みなら無視

    try {
      // 入力フィールドに値があるかチェック（結果表示の判定）
      let hasInputValues = false;
      INPUT_FIELDS.forEach(fieldName => {
        const input = document.querySelector(`input[name="${fieldName}"]`);
        if (input && input.value && input.value.trim()) {
          hasInputValues = true;
        }
      });

      if (hasInputValues) {
        hasExtracted = true;
        observer.disconnect();

        // DOMが完全に更新されるのを待ってから抽出
        setTimeout(() => {
          extractAndSendAllStatuses();
        }, DELAYS.EXTRACTION_DELAY);
      }
    } catch (error) {
      console.error('MutationObserver callback error:', error);
      hasExtracted = true;
      observer.disconnect();
    }
  });

  // body要素の変化を監視
  try {
    observer.observe(document.body, {
      childList: true,
      subtree: true
    });

    // タイムアウト処理
    setTimeout(() => {
      if (!hasExtracted) {
        hasExtracted = true;
        observer.disconnect();

        // タイムアウト後も入力フィールドに値がある場合は最終試行
        try {
          let hasInputValues = false;
          INPUT_FIELDS.forEach(fieldName => {
            const input = document.querySelector(`input[name="${fieldName}"]`);
            if (input && input.value && input.value.trim()) {
              hasInputValues = true;
            }
          });

          if (hasInputValues) {
            extractAndSendAllStatuses();
          }
        } catch (error) {
          console.error('Timeout extraction error:', error);
        }
      }
    }, DELAYS.RESULT_PAGE_OBSERVE_TIMEOUT); // 25秒タイムアウト（西濃サイトが遅いため延長）
  } catch (error) {
    console.error('Observer setup error:', error);
    observer.disconnect();
  }
}

// 重複実行防止フラグ
let isExtracting = false;

// 結果ページで全てのステータスを抽出してBackgroundに送信
function extractAndSendAllStatuses() {
  if (isExtracting) {
    return;
  }

  isExtracting = true;

  // 少し待機してページが完全に読み込まれるのを待つ
  setTimeout(() => {
    try {
      const results = [];

      // テーブルから各伝票番号とそのステータスを個別に取得
      const tables = document.querySelectorAll('table');

      tables.forEach((table, tableIndex) => {
        const rows = table.querySelectorAll('tr');

        rows.forEach((row, rowIndex) => {
          const rowText = row.textContent;
          const cells = row.querySelectorAll('td');

          // 行内に10-12桁の伝票番号があるか確認
          const numberMatch = rowText.match(/\b(\d{10,12})\b/);

          if (numberMatch && isValidTrackingNumber(numberMatch[1])) {
            const number = numberMatch[1];

            // お届け状況を抽出（実際のメッセージをそのまま取得）
            let status = '不明';
            let statusType = 'unknown';

            // エラーメッセージのチェック
            if (rowText.includes('見当りません') || rowText.includes('見当たりません')) {
              status = '入力されたお問合せ番号が見当りません';
              statusType = 'error';
            } else if (rowText.includes('配達済み')) {
              status = '配達済みです';
              statusType = 'delivered';
            } else if (rowText.includes('配達中') || rowText.includes('持出中')) {
              status = '配達中です';
              statusType = 'intransit';
            } else if (rowText.includes('輸送中')) {
              status = '輸送中です';
              statusType = 'intransit';
            } else if (rowText.includes('到着')) {
              status = '到着しました';
              statusType = 'delivered';
            } else if (rowText.includes('発送') || rowText.includes('出荷')) {
              status = '発送済みです';
              statusType = 'shipped';
            } else if (rowText.includes('受付') || rowText.includes('集荷')) {
              status = '受付済みです';
              statusType = 'pickedup';
            }

            // 重複チェック（同じ番号は追加しない）
            if (!results.find(r => r.number === number)) {
              results.push({
                number: number,
                status: status,
                statusType: statusType,
                dates: {},
                url: window.location.href,
                timestamp: new Date().toISOString()
              });
            }
          }
        });
      });

      // テーブルから取得できなかった場合は入力フィールドからフォールバック
      if (results.length === 0) {
        const pageText = document.body.textContent;

        INPUT_FIELDS.forEach(fieldName => {
          const input = document.querySelector(`input[name="${fieldName}"]`);
          if (input && input.value) {
            const number = input.value.trim();
            let status = '不明';
            let statusType = 'unknown';

            if (pageText.includes('見当りません') || pageText.includes('見当たりません')) {
              status = '番号なし';
              statusType = 'error';
            } else {
              // ステータスを抽出
              if (pageText.includes('完了') || pageText.includes('お届け済み')) {
                status = '完了';
              } else if (pageText.includes('到着')) {
                status = '到着';
              } else if (pageText.includes('持出')) {
                status = '持出';
              } else if (pageText.includes('発送') || pageText.includes('出荷')) {
                status = '発送';
              } else if (pageText.includes('受付') || pageText.includes('集荷')) {
                status = '受付';
              } else {
                status = '不明';
              }
              statusType = getStatusType(status);
            }

            results.push({
              number: number,
              status: status,
              statusType: statusType,
              dates: {},
              url: window.location.href,
              timestamp: new Date().toISOString()
            });
          }
        });
      }

      if (results.length > 0) {
        // Backgroundに結果を送信
        chrome.runtime.sendMessage({
          action: 'trackingResults',
          data: results
        }).then(() => {
          // 送信完了
        }).catch(err => {
          // エラー無視
        });
      } else {
        // 結果が0件の場合も空配列を送信
        chrome.runtime.sendMessage({
          action: 'trackingResults',
          data: []
        }).catch(err => {
          // エラー無視
        });
      }

    } catch (error) {

      // エラーも送信
      chrome.runtime.sendMessage({
        action: 'trackingResults',
        data: [{
          number: '不明',
          status: 'エラー',
          dates: {},
          url: window.location.href,
          error: error.message
        }]
      }).catch(err => {
        // エラー無視
      });
    } finally {
      // フラグをリセット
      isExtracting = false;
    }
  }, DELAYS.EXTRACTION_DELAY); // 待機時間を2秒に延長
}

// ページ読み込み時に結果ページかどうかを判定し、自動的に結果を抽出
setTimeout(() => {
  // ページから伝票番号を抽出してみて、あれば結果ページと判定
  const numbers = extractNumbersFromPage();

  if (numbers.length > 0) {
    extractAndSendAllStatuses();
  }
}, DELAYS.PAGE_LOAD_DELAY); // ページ読み込み後2秒待機してから判定

// テスト用エクスポート（本番環境では無視される）
if (typeof exports !== 'undefined') {
  exports.checkIsResultPage = checkIsResultPage;
  exports.extractNumbersFromPage = extractNumbersFromPage;
  exports.trackNumbersAndSubmit = trackNumbersAndSubmit;
  exports.extractAndSendAllStatuses = extractAndSendAllStatuses;
  exports.observeResultPage = observeResultPage;
  // common.jsの関数もエクスポート（テスト用）
  exports.getStatusType = getStatusType;
  exports.isValidTrackingNumber = isValidTrackingNumber;
}
