/**
 * 西濃追跡チェッカー - Background Service Worker テスト
 */

import { describe, it, expect, vi, beforeEach } from 'vitest';

// common.jsのモック
vi.mock('../utils/common.js', () => ({
  extractTrackingNumbers: vi.fn(),
  getStatusType: vi.fn()
}));

// Chrome API モックの作成
const mockTabs = {
  create: vi.fn(),
  update: vi.fn(),
  get: vi.fn(),
  sendMessage: vi.fn(),
  query: vi.fn(),
  onUpdated: {
    addListener: vi.fn(),
    removeListener: vi.fn()
  }
};

const mockStorage = {
  local: {
    get: vi.fn(),
    set: vi.fn(),
    remove: vi.fn()
  }
};

const mockRuntime = {
  sendMessage: vi.fn(),
  onMessage: {
    addListener: vi.fn(),
    removeListener: vi.fn()
  },
  onInstalled: {
    addListener: vi.fn()
  },
  onStartup: {
    addListener: vi.fn()
  },
  getURL: vi.fn((path) => `chrome-extension://test/${path}`)
};

const mockAction = {
  setBadgeText: vi.fn(),
  setBadgeBackgroundColor: vi.fn()
};

const mockContextMenus = {
  create: vi.fn(),
  removeAll: vi.fn(),
  onClicked: {
    addListener: vi.fn()
  }
};

const mockNotifications = {
  create: vi.fn()
};

global.chrome = {
  tabs: mockTabs,
  storage: mockStorage,
  runtime: mockRuntime,
  action: mockAction,
  contextMenus: mockContextMenus,
  notifications: mockNotifications
};

// モックのリセット
beforeEach(() => {
  vi.clearAllMocks();

  // デフォルトのモック戻り値を設定
  mockTabs.create.mockResolvedValue({
    id: 123,
    url: 'https://track.seino.co.jp/kamotsu/GempyoNoShokai.do',
    status: 'complete'
  });

  mockTabs.get.mockResolvedValue({
    id: 123,
    url: 'https://track.seino.co.jp/kamotsu/GempyoNoShokai.do',
    status: 'complete'
  });

  mockTabs.update.mockResolvedValue({
    id: 123,
    active: true
  });

  mockTabs.sendMessage.mockResolvedValue({
    success: true
  });

  mockStorage.local.get.mockResolvedValue({});
  mockStorage.local.set.mockResolvedValue(undefined);
  mockStorage.local.remove.mockResolvedValue(undefined);

  mockRuntime.getURL.mockImplementation((path) => `chrome-extension://test/${path}`);
});

describe('Background Service Worker', () => {
  describe('Chrome API モック検証', () => {
    it('chrome オブジェクトが正しくモックされている', () => {
      expect(global.chrome).toBeDefined();
      expect(global.chrome.tabs).toBeDefined();
      expect(global.chrome.storage).toBeDefined();
      expect(global.chrome.runtime).toBeDefined();
      expect(global.chrome.action).toBeDefined();
      expect(global.chrome.contextMenus).toBeDefined();
    });

    it('chrome.tabs メソッドがモックされている', () => {
      expect(typeof mockTabs.create).toBe('function');
      expect(typeof mockTabs.update).toBe('function');
      expect(typeof mockTabs.get).toBe('function');
      expect(typeof mockTabs.sendMessage).toBe('function');
    });

    it('chrome.storage.local メソッドがモックされている', () => {
      expect(typeof mockStorage.local.get).toBe('function');
      expect(typeof mockStorage.local.set).toBe('function');
      expect(typeof mockStorage.local.remove).toBe('function');
    });

    it('chrome.action メソッドがモックされている', () => {
      expect(typeof mockAction.setBadgeText).toBe('function');
      expect(typeof mockAction.setBadgeBackgroundColor).toBe('function');
    });
  });

  describe('バッチ処理ロジック', () => {
    it('10個以下の伝票番号は1つのバッチで処理される', () => {
      const numbers = Array.from({ length: 5 }, (_, i) => `${1234567890 + i}`);
      const MAX_PER_BATCH = 10;

      const batches = [];
      for (let i = 0; i < numbers.length; i += MAX_PER_BATCH) {
        batches.push(numbers.slice(i, i + MAX_PER_BATCH));
      }

      expect(batches.length).toBe(1);
      expect(batches[0].length).toBe(5);
    });

    it('20個の伝票番号は2つのバッチに分割される', () => {
      const numbers = Array.from({ length: 20 }, (_, i) => `${1234567890 + i}`);
      const MAX_PER_BATCH = 10;

      const batches = [];
      for (let i = 0; i < numbers.length; i += MAX_PER_BATCH) {
        batches.push(numbers.slice(i, i + MAX_PER_BATCH));
      }

      expect(batches.length).toBe(2);
      expect(batches[0].length).toBe(10);
      expect(batches[1].length).toBe(10);
    });

    it('25個の伝票番号は3つのバッチに分割される', () => {
      const numbers = Array.from({ length: 25 }, (_, i) => `${1234567890 + i}`);
      const MAX_PER_BATCH = 10;

      const batches = [];
      for (let i = 0; i < numbers.length; i += MAX_PER_BATCH) {
        batches.push(numbers.slice(i, i + MAX_PER_BATCH));
      }

      expect(batches.length).toBe(3);
      expect(batches[0].length).toBe(10);
      expect(batches[1].length).toBe(10);
      expect(batches[2].length).toBe(5);
    });
  });

  describe('進捗管理', () => {
    it('バッジテキストのフォーマットを検証', () => {
      const total = 10;
      const completed = 5;

      const badgeText = `${completed}/${total}`;
      expect(badgeText).toBe('5/10');
    });

    it('進捗保存オブジェクトの構造を検証', () => {
      const progress = {
        total: 10,
        completed: 5,
        startTime: new Date().toISOString()
      };

      expect(progress).toHaveProperty('total');
      expect(progress).toHaveProperty('completed');
      expect(progress).toHaveProperty('startTime');
      expect(typeof progress.startTime).toBe('string');
    });

    it('完了時にバッジが空になる', () => {
      const badgeText = '';
      expect(badgeText).toBe('');
    });
  });

  describe('エラーハンドリング', () => {
    it('エラー結果のフォーマットを検証', () => {
      const errorResult = {
        number: '1234567890',
        status: 'エラー',
        statusType: 'error',
        dates: {},
        url: 'https://track.seino.co.jp/?no=1234567890',
        error: 'テストエラー',
        timestamp: new Date().toISOString()
      };

      expect(errorResult).toHaveProperty('number');
      expect(errorResult).toHaveProperty('status');
      expect(errorResult).toHaveProperty('statusType');
      expect(errorResult).toHaveProperty('dates');
      expect(errorResult).toHaveProperty('url');
      expect(errorResult).toHaveProperty('error');
      expect(errorResult).toHaveProperty('timestamp');
      expect(errorResult.status).toBe('エラー');
      expect(errorResult.statusType).toBe('error');
    });

    it('タブが閉じられた場合のエラーメッセージを検証', () => {
      const errorMessage = 'タブが閉じられました';
      expect(errorMessage).toBe('タブが閉じられました');
    });

    it('タイムアウトのエラーメッセージを検証', () => {
      const errorMessage = '結果の取得がタイムアウトしました';
      expect(errorMessage).toBe('結果の取得がタイムアウトしました');
    });
  });

  describe('結果保存', () => {
    it('結果オブジェクトの構造を検証', () => {
      const result = {
        number: '1234567890',
        status: '配達完了',
        statusType: 'delivered',
        dates: {
          '受付': '2024-01-01 10:00',
          '配達完了': '2024-01-02 15:30'
        },
        url: 'https://track.seino.co.jp/?no=1234567890',
        timestamp: new Date().toISOString()
      };

      expect(result).toHaveProperty('number');
      expect(result).toHaveProperty('status');
      expect(result).toHaveProperty('statusType');
      expect(result).toHaveProperty('dates');
      expect(result).toHaveProperty('url');
      expect(result).toHaveProperty('timestamp');
      expect(typeof result.dates).toBe('object');
    });

    it('storage保存のフォーマットを検証', () => {
      const results = [
        {
          number: '1234567890',
          status: '配達完了',
          statusType: 'delivered',
          dates: {},
          url: 'https://track.seino.co.jp/?no=1234567890',
          timestamp: new Date().toISOString()
        }
      ];

      const storageData = { latestResults: results };

      expect(storageData).toHaveProperty('latestResults');
      expect(Array.isArray(storageData.latestResults)).toBe(true);
      expect(storageData.latestResults.length).toBe(1);
    });
  });

  describe('Chrome Events', () => {
    it('onInstalledリスナーが登録されている', () => {
      expect(typeof mockRuntime.onInstalled.addListener).toBe('function');
    });

    it('onStartupリスナーが登録されている', () => {
      expect(typeof mockRuntime.onStartup.addListener).toBe('function');
    });

    it('onClickedリスナーが登録されている', () => {
      expect(typeof mockContextMenus.onClicked.addListener).toBe('function');
    });

    it('onMessageリスナーが登録されている', () => {
      expect(typeof mockRuntime.onMessage.addListener).toBe('function');
    });
  });

  describe('Edge Cases', () => {
    it('空の配列が渡された場合に何もしない', () => {
      const numbers = [];
      const MAX_PER_BATCH = 10;

      const batches = [];
      for (let i = 0; i < numbers.length; i += MAX_PER_BATCH) {
        batches.push(numbers.slice(i, i + MAX_PER_BATCH));
      }

      expect(batches.length).toBe(0);
    });

    it('1つの伝票番号のみの場合に正しく処理される', () => {
      const numbers = ['1234567890'];
      const MAX_PER_BATCH = 10;

      const batches = [];
      for (let i = 0; i < numbers.length; i += MAX_PER_BATCH) {
        batches.push(numbers.slice(i, i + MAX_PER_BATCH));
      }

      expect(batches.length).toBe(1);
      expect(batches[0]).toEqual(['1234567890']);
    });

    it('正確に10個の伝票番号の場合に1つのバッチになる', () => {
      const numbers = Array.from({ length: 10 }, (_, i) => `${1234567890 + i}`);
      const MAX_PER_BATCH = 10;

      const batches = [];
      for (let i = 0; i < numbers.length; i += MAX_PER_BATCH) {
        batches.push(numbers.slice(i, i + MAX_PER_BATCH));
      }

      expect(batches.length).toBe(1);
      expect(batches[0].length).toBe(10);
    });

    it('11個の伝票番号の場合に2つのバッチになる', () => {
      const numbers = Array.from({ length: 11 }, (_, i) => `${1234567890 + i}`);
      const MAX_PER_BATCH = 10;

      const batches = [];
      for (let i = 0; i < numbers.length; i += MAX_PER_BATCH) {
        batches.push(numbers.slice(i, i + MAX_PER_BATCH));
      }

      expect(batches.length).toBe(2);
      expect(batches[0].length).toBe(10);
      expect(batches[1].length).toBe(1);
    });
  });

  describe('Performance', () => {
    it('大量の伝票番号を効率的に処理する', () => {
      const numbers = Array.from({ length: 100 }, (_, i) => `${1234567890 + i}`);
      const MAX_PER_BATCH = 10;

      const batches = [];
      for (let i = 0; i < numbers.length; i += MAX_PER_BATCH) {
        batches.push(numbers.slice(i, i + MAX_PER_BATCH));
      }

      expect(batches.length).toBe(10);

      batches.forEach(batch => {
        expect(batch.length).toBeLessThanOrEqual(10);
      });
    });

    it('進捗更新の頻度を検証', () => {
      const numbers = Array.from({ length: 25 }, (_, i) => `${1234567890 + i}`);
      const MAX_PER_BATCH = 10;

      const batchCount = Math.ceil(numbers.length / MAX_PER_BATCH);

      expect(batchCount).toBe(3);
    });
  });

  describe('定数', () => {
    it('DELAYS定数が正しく定義されている', () => {
      const DELAYS = {
        CONTENT_SCRIPT_READY: 3000,
        TAB_COMPLETE: 1500,
        CONTENT_SCRIPT_TIMEOUT: 5000,
        RESULT_TIMEOUT: 30000
      };

      expect(DELAYS.CONTENT_SCRIPT_READY).toBe(3000);
      expect(DELAYS.TAB_COMPLETE).toBe(1500);
      expect(DELAYS.CONTENT_SCRIPT_TIMEOUT).toBe(5000);
      expect(DELAYS.RESULT_TIMEOUT).toBe(30000);
    });

    it('MAX_PER_BATCH定数が正しく定義されている', () => {
      const MAX_PER_BATCH = 10;
      expect(MAX_PER_BATCH).toBe(10);
    });

    it('MAX_HISTORY_SIZE定数が正しく定義されている', () => {
      const MAX_HISTORY_SIZE = 50;
      expect(MAX_HISTORY_SIZE).toBe(50);
    });

    it('STORAGE_KEY_PROGRESS定数が正しく定義されている', () => {
      const STORAGE_KEY_PROGRESS = 'trackingProgress';
      expect(STORAGE_KEY_PROGRESS).toBe('trackingProgress');
    });
  });

  describe('URL生成', () => {
    it('西濃運輸の追跡ページURLを生成する', () => {
      const trackingUrl = 'https://track.seino.co.jp/kamotsu/GempyoNoShokai.do';
      expect(trackingUrl).toBe('https://track.seino.co.jp/kamotsu/GempyoNoShokai.do');
    });

    it('伝票番号付きのURLを生成する', () => {
      const number = '1234567890';
      const url = `https://track.seino.co.jp/?no=${number}`;
      expect(url).toBe('https://track.seino.co.jp/?no=1234567890');
    });
  });

  describe('Chrome API メソッド呼び出し', () => {
    it('chrome.tabs.createが正しく呼び出される', async () => {
      mockTabs.create.mockResolvedValue({
        id: 123,
        url: 'https://track.seino.co.jp/kamotsu/GempyoNoShokai.do'
      });

      const tab = await mockTabs.create({
        url: 'https://track.seino.co.jp/kamotsu/GempyoNoShokai.do',
        active: false
      });

      expect(mockTabs.create).toHaveBeenCalledWith({
        url: 'https://track.seino.co.jp/kamotsu/GempyoNoShokai.do',
        active: false
      });
      expect(tab.id).toBe(123);
    });

    it('chrome.tabs.updateが正しく呼び出される', async () => {
      mockTabs.update.mockResolvedValue({
        id: 123,
        active: true
      });

      const tab = await mockTabs.update(123, { active: true });

      expect(mockTabs.update).toHaveBeenCalledWith(123, { active: true });
      expect(tab.active).toBe(true);
    });

    it('chrome.storage.local.setが正しく呼び出される', async () => {
      mockStorage.local.set.mockResolvedValue(undefined);

      await mockStorage.local.set({ trackingProgress: { total: 10, completed: 5 } });

      expect(mockStorage.local.set).toHaveBeenCalledWith({
        trackingProgress: { total: 10, completed: 5 }
      });
    });

    it('chrome.action.setBadgeTextが正しく呼び出される', () => {
      mockAction.setBadgeText({ text: '5/10' });

      expect(mockAction.setBadgeText).toHaveBeenCalledWith({ text: '5/10' });
    });

    it('chrome.action.setBadgeBackgroundColorが正しく呼び出される', () => {
      mockAction.setBadgeBackgroundColor({ color: '#4CAF50' });

      expect(mockAction.setBadgeBackgroundColor).toHaveBeenCalledWith({ color: '#4CAF50' });
    });
  });

  describe('タイムアウト処理', () => {
    it('CONTENT_SCRIPT_TIMEOUTでタイムアウトする', () => {
      const CONTENT_SCRIPT_TIMEOUT = 5000;
      expect(CONTENT_SCRIPT_TIMEOUT).toBe(5000);
    });

    it('RESULT_TIMEOUTでタイムアウトする', () => {
      const RESULT_TIMEOUT = 30000;
      expect(RESULT_TIMEOUT).toBe(30000);
    });

    it('TAB_COMPLETEの遅延時間を検証', () => {
      const TAB_COMPLETE = 1500;
      expect(TAB_COMPLETE).toBe(1500);
    });

    it('CONTENT_SCRIPT_READYの遅延時間を検証', () => {
      const CONTENT_SCRIPT_READY = 3000;
      expect(CONTENT_SCRIPT_READY).toBe(3000);
    });
  });

  describe('バッジカラー', () => {
    it('進行中のバッジカラーを検証', () => {
      const progressColor = '#4CAF50';
      expect(progressColor).toBe('#4CAF50');
    });

    it('進捗復元時のバッジカラーを検証', () => {
      const resumeColor = '#FF9800';
      expect(resumeColor).toBe('#FF9800');
    });
  });

  describe('ストレージキー', () => {
    it('進捗保存のストレージキーを検証', () => {
      const STORAGE_KEY_PROGRESS = 'trackingProgress';
      const storageKey = `trackingProgress`;

      expect(storageKey).toBe(STORAGE_KEY_PROGRESS);
    });

    it('結果保存のストレージキーを検証', () => {
      const resultsKey = 'latestResults';
      expect(resultsKey).toBe('latestResults');
    });
  });
});
