/**
 * 西濃追跡チェッカー - Background Service Worker
 *
 * 機能:
 * - コンテキストメニューの作成・管理
 * - 伝票番号の抽出・解析
 * - 西濃運輸ページを開いてContent Scriptにメッセージ送信
 * - 結果のStorage保存とポップアップ表示
 */

// Service WorkerではimportScripts()を使用してモジュールを読み込む
importScripts('../utils/common.js');

// 定数
const DELAYS = {
  CONTENT_SCRIPT_READY: 3000,
  TAB_COMPLETE: 1500,
  CONTENT_SCRIPT_TIMEOUT: 5000,
  RESULT_TIMEOUT: 30000
};

const MAX_PER_BATCH = 10;
const MAX_HISTORY_SIZE = 50;
const STORAGE_KEY_PROGRESS = 'trackingProgress';

// タブIDごとのPromise resolverを管理
const pendingResults = new Map();

// 進捗を保存
async function saveProgress(progress) {
  await chrome.storage.local.set({ [STORAGE_KEY_PROGRESS]: progress });
}

// 進捗を復元
async function loadProgress() {
  const result = await chrome.storage.local.get([STORAGE_KEY_PROGRESS]);
  return result[STORAGE_KEY_PROGRESS] || null;
}

// 進捗をクリア
async function clearProgress() {
  await chrome.storage.local.remove([STORAGE_KEY_PROGRESS]);
}

// 伝票番号を追跡
async function trackNumbers(numbers, senderTab) {
  const results = [];

  // MAX_PER_BATCH個ずつグループ化（西濃運輸の入力フィールドは最大10個）
  const batches = [];
  for (let i = 0; i < numbers.length; i += MAX_PER_BATCH) {
    batches.push(numbers.slice(i, i + MAX_PER_BATCH));
  }

  // 進捗表示
  chrome.action.setBadgeText({ text: `0/${numbers.length}` });
  chrome.action.setBadgeBackgroundColor({ color: '#4CAF50' });

  // 進捗保存
  await saveProgress({
    total: numbers.length,
    completed: 0,
    startTime: new Date().toISOString()
  });

  // バッチ処理
  for (let batchIndex = 0; batchIndex < batches.length; batchIndex++) {
    const batch = batches[batchIndex];

    // タブを作成
    const tab = await chrome.tabs.create({
      url: 'https://track.seino.co.jp/kamotsu/GempyoNoShokai.do',
      active: false  // バックグラウンドで開く
    });

    // ページ読み込み完了を待機
    await waitForTabComplete(tab.id);

    // 追加: Content Scriptの確実なロードを待つ
    await new Promise(resolve => setTimeout(resolve, DELAYS.CONTENT_SCRIPT_READY));

    // タブがまだ存在するか確認
    const tabExists = await chrome.tabs.get(tab.id).catch((error) => {
      console.error('Error checking tab existence:', error);
      return null;
    });
    if (!tabExists) {
      batch.forEach(number => {
        results.push({
          number: number,
          status: 'エラー',
          statusType: 'error',
          dates: {},
          url: `https://track.seino.co.jp/?no=${number}`,
          error: 'タブが閉じられました',
          timestamp: new Date().toISOString()
        });
      });
      continue; // 次のバッチへ
    }

    await chrome.tabs.update(tab.id, { active: true });

    // Content Scriptの準備完了を待つ
    await waitForContentScriptReady(tab.id);

    // Content Scriptにメッセージ送信（バッチ全体を送信）
    try {
      const response = await chrome.tabs.sendMessage(tab.id, {
        action: 'trackNumbers',
        numbers: batch
      });

      // 結果ページからのレスポンスを待機
      const batchResults = await waitForResult(tab.id, batch);

      if (batchResults && batchResults.length > 0) {
        results.push(...batchResults);
      } else {
        // タイムアウトの場合は全てエラーとして扱う
        batch.forEach(number => {
          results.push({
            number: number,
            status: 'エラー',
            statusType: 'error',
            dates: {},
            url: `https://track.seino.co.jp/?no=${number}`,
            error: '結果の取得がタイムアウトしました',
            timestamp: new Date().toISOString()
          });
        });
      }
    } catch (error) {
      batch.forEach(number => {
        results.push({
          number: number,
          status: 'エラー',
          statusType: 'error',
          dates: {},
          url: `https://track.seino.co.jp/?no=${number}`,
          error: error.message,
          timestamp: new Date().toISOString()
        });
      });
    }

    // 進捗更新
    const completedCount = Math.min((batchIndex + 1) * MAX_PER_BATCH, numbers.length);
    chrome.action.setBadgeText({ text: `${completedCount}/${numbers.length}` });

    // 進捗保存
    await saveProgress({
      total: numbers.length,
      completed: completedCount,
      startTime: new Date().toISOString()
    });

    // 次のバッチがある場合は新しいタブを作成（現在のタブはそのまま）
  }

  // 進捗をクリア
  await clearProgress();
  chrome.action.setBadgeText({ text: '' });

  // 結果をStorageに保存
  await chrome.storage.local.set({ latestResults: results });

  // ポップアップは自動で開かない（ユーザーが手動で開く）

  return results;
}

// 結果を待機（複数の結果に対応）
function waitForResult(tabId, numbers) {
  return new Promise((resolve) => {
    const timeout = setTimeout(() => {
      pendingResults.delete(tabId);
      resolve(null);
    }, DELAYS.RESULT_TIMEOUT);

    pendingResults.set(tabId, { resolve, timeout });
  });
}

// タブの読み込み完了を待機
function waitForTabComplete(tabId) {
  return new Promise((resolve) => {
    const listener = (updatedTabId, changeInfo) => {
      if (updatedTabId === tabId && changeInfo.status === 'complete') {
        chrome.tabs.onUpdated.removeListener(listener);
        // 追加の待機（JavaScript実行完了を待つ）
        setTimeout(resolve, DELAYS.TAB_COMPLETE);
      }
    };
    chrome.tabs.onUpdated.addListener(listener);
  });
}

// Content Scriptの準備完了を待機
function waitForContentScriptReady(tabId) {
  return new Promise((resolve) => {
    const timeout = setTimeout(() => {
      chrome.runtime.onMessage.removeListener(listener);
      resolve(); // タイムアウトでも進む
    }, DELAYS.CONTENT_SCRIPT_TIMEOUT);

    const listener = (request, sender) => {
      if (request.action === 'contentScriptReady') {
        clearTimeout(timeout);
        chrome.runtime.onMessage.removeListener(listener);
        resolve();
      }
    };

    chrome.runtime.onMessage.addListener(listener);
  });
}

// コンテキストメニュークリック時の処理
chrome.contextMenus.onClicked.addListener(async (info, tab) => {
  if (info.menuItemId === 'trackSeino') {
    // 選択テキストから伝票番号を抽出
    const numbers = extractTrackingNumbers(info.selectionText);

    if (numbers.length === 0) {
      // エラー通知
      await chrome.notifications.create({
        type: 'basic',
        iconUrl: chrome.runtime.getURL('icons/icon48.png'),
        title: '西濃追跡チェッカー',
        message: '伝票番号が見つかりませんでした'
      });
      return;
    }

    // 追跡開始
    await trackNumbers(numbers, tab);
  }
});

// 拡張機能インストール時にコンテキストメニューを作成
chrome.runtime.onInstalled.addListener(() => {
  chrome.contextMenus.removeAll(() => {
    chrome.contextMenus.create({
      id: 'trackSeino',
      title: '西濃運輸で追跡',
      contexts: ['selection']
    });
  });
});

// Service Worker起動時に進捗を復元
chrome.runtime.onStartup.addListener(async () => {
  const progress = await loadProgress();
  if (progress) {
    chrome.action.setBadgeText({
      text: `${progress.completed}/${progress.total}`
    });
    chrome.action.setBadgeBackgroundColor({ color: '#FF9800' }); // オレンジ（進行中）
  }
});

// ポップアップからのメッセージ受信
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'trackNumbers') {
    trackNumbers(request.numbers)
      .then(results => sendResponse({ success: true, results }))
      .catch(error => sendResponse({ success: false, error: error.message }));
    return true; // 非同期レスポンス
  }

  // trackingResults処理を追加
  if (request.action === 'trackingResults') {
    const pending = pendingResults.get(sender.tab.id);
    if (pending) {
      clearTimeout(pending.timeout);
      pendingResults.delete(sender.tab.id);
      pending.resolve(request.data);
    }
    return true;
  }
});

// テスト用エクスポート（本番環境では無視される）
if (typeof exports !== 'undefined') {
  exports.trackNumbers = trackNumbers;
  exports.waitForResult = waitForResult;
  exports.waitForTabComplete = waitForTabComplete;
  exports.waitForContentScriptReady = waitForContentScriptReady;
  exports.saveProgress = saveProgress;
  exports.loadProgress = loadProgress;
  exports.clearProgress = clearProgress;
}
